************************************************************************
*                                                                      *
*                            This is OLO_S4                            *
*                                                                      *
* for the evaluation of one-loop scalar 2-, 3- and 4-point functions   *
* with all internal masses equal zero.                                 *
*                                                                      *
* author: Andreas van Hameren <Andre.HamerenREMOVETHIS@ifj.edu.pl>     *
*   date: 20-11-2007                                                   *
*                                                                      *
* Please cite                                                          *
*    A. van Hameren, J. Vollinga, S. Weinzierl,                        *
*    Eur.Phys.J.C41:361-375,2005, (hep-ph/0502165)                     *
* in publications with results obtained with the help of this program. *
************************************************************************
*
* 2-point function: 
*   subroutine avh_olo_b0(valb0,p1)
*     input:  double precision p1 = squared external mass 
*     output: double complex valb0(0) = eps^0   -coefficient
*                            valb0(1) = eps^(-1)-coefficient
*                            valb0(2) = eps^(-2)-coefficient
*
* 3-point function: 
*   subroutine avh_olo_c0(valc0,p1,p2,p3)
*     input:  double precision p1, p2, p3 = squared external masses 
*     output: double complex valc0(0) = eps^0   -coefficient
*                            valc0(1) = eps^(-1)-coefficient
*                            valc0(2) = eps^(-2)-coefficient
*
* 4-point function: 
*   subroutine avh_olo_d0(vald0,p1,p2,p3,p4,p12,p23)
*     input:  double precision p1, p2, p3, p4 = squared external masses
*                              p12=(k1+k2)^2, p23=(k2+k3)^2
*     output: double complex vald0(0) = eps^0   -coefficient
*                            vald0(1) = eps^(-1)-coefficient
*                            vald0(2) = eps^(-2)-coefficient
*
* to set the renormalization scale:
*    subroutine avh_olo_mu_set(mu)
*      input: double precision mu , has unit of mass (so is not mu^2)
*    If this routine is not called, mu is set to the default mu=1d0.
*
* to get the renormalization scale:
*    function avh_olo_mu_get()
*      output: double precision avh_olo_mu_get() = mu
*
* Check the comments in the routines themselves for more details.
*
* IMPORTANT: IR-singular cases are returned only if external
*            masses are identically 0d0.
*
* Based on the work of G. Duplancic and B. Nizic, 
*                      Eur.Phys.J.C20:357-370,2001 (hep-ph/0006249)
*                      Eur.Phys.J.C24:385-391,2002 (hep-ph/0201306)
*
***********************************************************************

      subroutine avh_olo_mu_set(mu_in)
*  ********************************************************************
*  ********************************************************************
      implicit none
      double precision mu_in
     &,mu ,avh_olo_mu_get
      common/avh_olo_mu_com/ mu
      integer
     & init
      data init/0/
      save init
*
      if (init.eq.0) then
        init = 1
        mu = avh_olo_mu_get()
      endif
      mu = mu_in
      end
*
      function avh_olo_mu_get()
*  ********************************************************************
*  ********************************************************************
      implicit none
      double precision avh_olo_mu_get
     &,mu
      common/avh_olo_mu_com/ mu
      integer
     & init
      data init/0/
      save init
*
      if (init.eq.0) then
        init = 1
        mu = 1d0
      endif
      avh_olo_mu_get = mu
      end

      subroutine avh_olo_d0(vald0,p1,p2,p3,p4,p12,p23)
*  ********************************************************************
*  * calculates
*  *               C   /              d^(Dim)q
*  *            ------ | ---------------------------------------
*  *            i*pi^2 / q^2 (q+k1)^2 (q+k1+k2)^2 (q+k1+k2+k3)^2
*  *
*  * with  Dim = 4-2*eps
*  *         C = pi^eps * mu^(2*eps) * exp(gamma_Euler*eps)
*  *
*  * input:  p1 = k1^2,  p2 = k2^2,  p3 = k3^2,  p4 = (k1+k2+k3)^2,
*  *         p12 = (k1+k2)^2,  p23 = (k2+k3)^2
*  * output: vald0(0) = eps^0   -coefficient
*  *         vald0(1) = eps^(-1)-coefficient
*  *         vald0(2) = eps^(-2)-coefficient
*  *
*  * The input values (p1,p2,p3,p4,p12,p23) should be real.
*  * If any of these numbers is IDENTICALLY 0d0, the corresponding
*  * IR-singular case is returned.
*  ********************************************************************
      implicit none
      double complex vald0(0:2)
     &,pi2,aa,bb,cc,log2,log3,log4,log5,log6,li22,li23,li24,li25,li26
     &,li254,li262,li263,avh_olo_log,avh_olo_li2,avh_olo_log2
      double precision p1,p2,p3,p4,p12,p23
     &,pi,mu2,pp(6),gg,ff,thrs,avh_olo_mu_get,avh_olo_pi,avh_olo_prec
     &,precfac
      integer
     & init,base(4),icase,ii,per(6),sf,sgn,imax
      character*3 label(6)
      data init/0/,base/8,4,2,1/,label/'p1','p2','p3','p4','p12','p23'/
      save init,pi2,thrs
*
      parameter(precfac=1d0)
*
      if (init.eq.0) then
        init = 1
        pi2  = dcmplx( avh_olo_pi()**2 )
        thrs = precfac*avh_olo_prec()
      endif
      mu2 = avh_olo_mu_get()**2
*
      pp(1) = p1
      pp(2) = p2
      pp(3) = p3
      pp(4) = p4
      pp(5) = p12
      pp(6) = p23
      ff = dabs(pp(1))
      imax = 1
      do ii=2,6
        gg = dabs(pp(ii))
        if (gg.gt.ff) then
          ff = gg
          imax = ii
        endif
      enddo
      ff = ff*thrs
      icase = 0
      do ii=1,4
      if (pp(ii).ne.0d0) then
        icase = icase + base(ii)
        gg = dabs(pp(ii))
        if (gg.lt.ff) write(6,*) 'WARNING from avh_olo_d0: '
     &    ,'|',label(ii),'/',label(imax),'| =',gg/ff*thrs
      endif
      enddo
      call avh_olo_d0per(icase,per)
*
      if     (icase.eq.0) then
* 0 masses non-zero
        gg = 1d0/( pp(per(5)) * pp(per(6)) )
        log5 = avh_olo_log(-pp(per(5))/mu2,-1 ,1)
        log6 = avh_olo_log(-pp(per(6))/mu2,-1 ,2)
        aa = dcmplx(4d0)
        bb = -2*(log5 + log6)
        cc = log5**2 + log6**2 
     &     - avh_olo_log2(pp(per(5)),1,pp(per(6)),1)**2 - 4*pi2/3
      elseif (icase.eq.1) then
* 1 mass non-zero
        gg = 1d0/( pp(per(5)) * pp(per(6)) )
        ff =  gg*( pp(per(5)) + pp(per(6)) - pp(per(4)) )
        sf = 1
        if (ff.lt.0d0) sf = -1
        log4 = avh_olo_log(-pp(per(4))/mu2,-1 ,3)
        log5 = avh_olo_log(-pp(per(5))/mu2,-1 ,4)
        log6 = avh_olo_log(-pp(per(6))/mu2,-1 ,5)
        li24 = avh_olo_li2( 1d0-pp(per(4))*ff , -sf )
        li25 = avh_olo_li2( 1d0-pp(per(5))*ff , -sf )
        li26 = avh_olo_li2( 1d0-pp(per(6))*ff , -sf )
        aa = dcmplx(2d0)
        bb = 2*(log4-log5-log6)
        cc = log5**2 + log6**2 - log4**2 
     &     - pi2/2 + 2*(li25 + li26 - li24)
      elseif (icase.eq.2) then
* 2 neighbour masses non-zero
        gg = 1d0/( pp(per(5)) * pp(per(6)) )
        ff =  gg*( pp(per(5)) + pp(per(6)) - pp(per(4)) )
        sf = 1
        if (ff.lt.0d0) sf = -1
        log3 = avh_olo_log(-pp(per(3))/mu2,-1 ,6)
        log4 = avh_olo_log(-pp(per(4))/mu2,-1 ,7)
        log5 = avh_olo_log(-pp(per(5))/mu2,-1 ,8)
        log6 = avh_olo_log(-pp(per(6))/mu2,-1 ,9)
        sgn = 1
        if (pp(per(5)).lt.pp(per(4))) sgn = -1
        li254 = avh_olo_li2( 1d0-pp(per(4))/pp(per(5)) , -sgn )
        sgn = 1
        if (pp(per(6)).lt.pp(per(3))) sgn = -1
        li263 = avh_olo_li2( 1d0-pp(per(3))/pp(per(6)) , -sgn )
        li24 = avh_olo_li2( 1d0-pp(per(4))*ff , -sf )
        li25 = avh_olo_li2( 1d0-pp(per(5))*ff , -sf )
        li26 = avh_olo_li2( 1d0-pp(per(6))*ff , -sf )
        aa = dcmplx(1d0)
        bb = log4 + log3 - log5 - 2*log6
        cc = log5**2 + log6**2 - log3**2 - log4**2
     &     +(log3 + log4 - log5)**2/2
     &     - pi2/12 + 2*(li254 - li263 + li25 + li26 - li24)
      elseif (icase.eq.5) then
* 2 opposite masses non-zero
        gg = 1d0/( pp(per(5)) * pp(per(6)) - pp(per(2)) * pp(per(4)) )
        ff =  gg*( pp(per(5)) + pp(per(6)) - pp(per(2)) - pp(per(4)) )
        sf = 1
        if (ff.lt.0d0) sf = -1
        log2 = avh_olo_log(-pp(per(2))/mu2,-1 ,10)
        log4 = avh_olo_log(-pp(per(4))/mu2,-1 ,11)
        log5 = avh_olo_log(-pp(per(5))/mu2,-1 ,12)
        log6 = avh_olo_log(-pp(per(6))/mu2,-1 ,13)
        li22 = avh_olo_li2( 1d0-pp(per(2))*ff , -sf )
        li24 = avh_olo_li2( 1d0-pp(per(4))*ff , -sf )
        li25 = avh_olo_li2( 1d0-pp(per(5))*ff , -sf )
        li26 = avh_olo_li2( 1d0-pp(per(6))*ff , -sf )
        aa = dcmplx(0d0)
        bb = 2*(log2 + log4 - log5 - log6)
        cc = log5**2 + log6**2 - log2**2 - log4**2 
     &     + 2*(li25 + li26 - li22 - li24)
      elseif (icase.eq.3) then
* 3 masses non-zero
        gg = 1d0/( pp(per(5)) * pp(per(6)) - pp(per(2)) * pp(per(4)) )
        ff =  gg*( pp(per(5)) + pp(per(6)) - pp(per(2)) - pp(per(4)) )
        sf = 1
        if (ff.lt.0d0) sf = -1
        log2 = avh_olo_log(-pp(per(2))/mu2,-1 ,14)
        log3 = avh_olo_log(-pp(per(3))/mu2,-1 ,15)
        log4 = avh_olo_log(-pp(per(4))/mu2,-1 ,16)
        log5 = avh_olo_log(-pp(per(5))/mu2,-1 ,17)
        log6 = avh_olo_log(-pp(per(6))/mu2,-1 ,18)
        li22 = avh_olo_li2( 1d0-pp(per(2))*ff , -sf )
        li24 = avh_olo_li2( 1d0-pp(per(4))*ff , -sf )
        li25 = avh_olo_li2( 1d0-pp(per(5))*ff , -sf )
        li26 = avh_olo_li2( 1d0-pp(per(6))*ff , -sf )
        sgn = 1
        if (pp(per(5)).lt.pp(per(4))) sgn = -1
        li254 = avh_olo_li2( 1d0-pp(per(4))/pp(per(5)) , -sgn )
        sgn = 1
        if (pp(per(6)).lt.pp(per(2))) sgn = -1
        li262 = avh_olo_li2( 1d0-pp(per(2))/pp(per(6)) , -sgn )
        aa = dcmplx(0d0)
        bb = log2 + log4 - log5 - log6
        cc = log5**2 + log6**2 - log2**2 - log3**2 - log4**2
     &     + (log3+log4-log5)**2/2 + (log3+log2-log6)**2/2
     &     + 2*(li262 + li254 + li25 + li26 - li22 - li24)
      elseif (icase.eq.4) then
* 4 masses non-zero
        gg = 1d0
        aa = dcmplx(0d0)
        bb = dcmplx(0d0)
        call avh_olo_d04m(cc,pp(per(1)),pp(per(2)),pp(per(3))
     &                       ,pp(per(4)),pp(per(5)),pp(per(6)) )
      endif
*
      vald0(0) = gg*cc
      vald0(1) = gg*bb
      vald0(2) = gg*aa
      end
*     
      subroutine avh_olo_d0per(icase,per)
*  ********************************************************************
*  * Go through all possibilities of zero (0) and non-zero (1) masses
*  *
*  *   mass: 1234     mass: 1234     mass: 1234     mass: 1234
*  * icase=1 0001  icase= 3 0011  icase= 7 0111  icase= 0 0000 icase->0
*  * icase=2 0010  icase= 6 0110  icase=14 1110  icase=15 1111 icase->4 
*  * icase=4 0100  icase=12 1100  icase=13 1101  icase= 5 0101 icase->5
*  * icase=8 1000  icase= 9 1001  icase=11 1011  icase=10 1010 icase->5
*  *   icase->1      icase->2       icase->3
*  ********************************************************************
      implicit none
      integer icase,per(6)
     &,permtable(6,0:15),casetable(0:15),ii
      data permtable/
     & 1,2,3,4 ,5,6 ! 0, 0 masses non-zero,           no perm
     &,1,2,3,4 ,5,6 ! 1, 1 mass non-zero,             no perm
     &,4,1,2,3 ,6,5 ! 2, 1 mass non-zero,             1 cyclic perm
     &,1,2,3,4 ,5,6 ! 3, 2 neighbour masses non-zero, no perm
     &,3,4,1,2 ,5,6 ! 4, 1 mass   non-zero,           2 cyclic perm's
     &,1,2,3,4 ,5,6 ! 5, 2 opposite masses non-zero,  no perm
     &,4,1,2,3 ,6,5 ! 6, 2 neighbour masses non-zero, 1 cyclic perm
     &,1,2,3,4 ,5,6 ! 7, 3 masses non-zero,           no perm
     &,2,3,4,1 ,6,5 ! 8, 1 mass   non-zero,           3 cyclic perm's
     &,2,3,4,1 ,6,5 ! 9, 2 neighbour masses non-zero, 3 cyclic perm's
     &,4,1,2,3 ,6,5 !10, 2 opposite masses non-zero,  1 cyclic perm
     &,2,3,4,1 ,6,5 !11, 3 masses non-zero,           3 cyclic perm's
     &,3,4,1,2 ,5,6 !12, 2 neighbour masses non-zero, 2 cyclic perm's
     &,3,4,1,2 ,5,6 !13, 3 masses non-zero,           2 cyclic perm's
     &,4,1,2,3 ,6,5 !14, 3 masses non-zero,           1 cyclic perm
     &,1,2,3,4 ,5,6 !15, 4 masses non-zero,           no perm
     &/             ! 0, 1, 2, 3, 4, 5, 6, 7, 8, 9,10,11,12,13,14,15
      data casetable/ 0, 1, 1, 2, 1, 5, 2, 3, 1, 2, 5, 3, 2, 3, 3, 4/
      do ii=1,6
        per(ii) = permtable(ii,icase)
      enddo
      icase = casetable(icase)
      end
*
      subroutine avh_olo_d04m(vald0,p1,p2,p3,p4,p12,p23)
*  ********************************************************************
*  ********************************************************************
      implicit none
      double complex vald0 
     &,clam,zz,valc0,avh_olo_log2
      double precision p1,p2,p3,p4,p12,p23
     &,pi,pp(3),rlam,alp(3),bet(3),xx(2),xc,ya,yb,r1,r2,r3,hh,avh_olo_pi
      integer
     & init,sgneps,sga(3),sgb(3),ii,sx1,sx2,mag(3),per(3)
      logical l_lt_0,l_ge_0
      data init/0/
      save init,pi
*
      if (init.eq.0) then
        init = 1
        pi = avh_olo_pi()
      endif
      alp(1) = p12
      bet(1) = p23
      alp(2) = p1
      bet(2) = p3
      alp(3) = p2
      bet(3) = p4
      sgneps = 0
      do ii=1,3
        sga(ii) = 1
        if (alp(ii).lt.0d0) sga(ii) = -1
        sgb(ii) = 1
        if (bet(ii).lt.0d0) sgb(ii) = -1
        sgneps = sgneps + sga(ii) + sgb(ii)
      enddo
      if (sgneps.lt.0) then
        sgneps = -1
        do ii=1,3
          alp(ii) = -alp(ii)
          sga(ii) = -sga(ii)
          bet(ii) = -bet(ii)
          sgb(ii) = -sgb(ii)
        enddo
      else
        sgneps = 1
      endif
      pp(1) = alp(1)*bet(1)
      pp(2) = alp(2)*bet(2)
      pp(3) = alp(3)*bet(3)
      call avh_olo_kallen(l_lt_0,rlam,mag ,pp)
      if (l_lt_0) then
        hh = pp(mag(1)) + pp(mag(2))
        hh = hh + pp(mag(3))
        if (hh.gt.0d0) then
          clam = dcmplx( 0d0, -sgneps*rlam )
        else
          clam = dcmplx( 0d0,  sgneps*rlam )
        endif
      else
        clam = dcmplx(rlam,0d0)
      endif
      vald0 = dcmplx(0d0)
      do ii=1,3
      if (sga(ii).lt.0.and.sgb(ii).lt.0) then
        per(3) = ii
        per(1) = ii+1
        if (per(1).gt.3) per(1) = per(1)-3
        per(2) = per(1)+1
        if (per(2).gt.3) per(2) = per(2)-3
        if (l_lt_0) then
          zz = dcmplx( pp(per(1)) + pp(per(2)) - pp(per(3)) ) + clam
          zz = dcmplx( 0d0 , 2*dimag(cdlog(zz)) )
        else
          sx1 = 1
          if (pp(per(3)).lt.rlam) sx1 = -1
          sx2 = 1
          if (pp(per(3)).lt.-rlam) sx2 = -1
          call avh_olo_d04m_h(xx ,rlam,pp,per,mag)
          zz = avh_olo_log2(xx(1),-sx1*sgneps ,xx(2),sx2*sgneps)
        endif
        vald0 = vald0 + zz
      endif  
      enddo
      vald0 = dcmplx(0d0,-2*sgneps*pi)*vald0/clam
      call avh_olo_c03m(valc0,pp(1),pp(2),pp(3),sgneps)
      vald0 = valc0 + vald0
      end

      subroutine avh_olo_d04m_h(xx ,lam,pp,per,mag)
*  ********************************************************************
*  * Calculate the solutions to the equation
*  *   x^2/2 - (p1+p2-p3)*x + 2*p1*p2 = 0
*  * given by
*  *   xx(1) = p1 + p2 - p3 + lam
*  *   xx(2) = p1 + p2 - p3 - lam
*  * where
*  *   lam = dsqrt( p1^2 + p2^2 + p3^2 - 2*(p1*p2 + p2*p3 + p3*p1) )
*  * for values of p1,p2,p3 such that lam is real.
*  ********************************************************************
      implicit none
      integer per(3),mag(3)
     &,iter,ii,niter
      double precision xx(2) ,lam,pp(3)
     &,hh,cc,bb,ff,gg,xo,thrs
*
      parameter(thrs=1d-2)
      parameter(niter=2)
*
      hh = pp(per(1)) + pp(per(2)) - pp(per(3))
      xx(1) = hh + lam
      xx(2) = hh - lam
*
* If an xx is close to zero, iterate Newton-Raphson a few times
      do ii=1,2
      if (dabs( xx(ii)/pp(mag(1)) ).lt.thrs) then
        cc = 2d0*pp(per(1))*pp(per(2))
        bb = -hh
        do iter=1,niter
          xo = xx(ii)
          ff = cc + bb*xo + xo*xo/2d0 
          gg = bb + 2d0*xo
          xx(ii) = xo - ff/gg
        enddo
      endif
      enddo
*
      end

 
      subroutine avh_olo_c0(valc0,p1,p2,p3)
*  ********************************************************************
*  * calculates
*  *               C   /         d^(Dim)q
*  *            ------ | ------------------------
*  *            i*pi^2 / q^2 (q+k1)^2 (q+k1+k2)^2
*  *
*  * with  Dim = 4-2*eps
*  *         C = pi^eps * mu^(2*eps) * exp(gamma_Euler*eps)
*  *
*  * input:  p1 = k1^2,  p2 = k2^2,  p3 = k3^2
*  * output: valc0(0) = eps^0   -coefficient
*  *         valc0(1) = eps^(-1)-coefficient
*  *         valc0(2) = eps^(-2)-coefficient
*  *
*  * The input values (p1,p2,p3) should be real.
*  * If any of these numbers is IDENTICALLY 0d0, the corresponding
*  * IR-singular case is returned.
*  ********************************************************************
      implicit none
      double complex valc0(0:2)
     &,pi2,log2,log3,avh_olo_log,avh_olo_log3
      double precision p1,p2,p3
     &,mu2,pp(3),ff,gg,thrs,avh_olo_mu_get,avh_olo_pi,avh_olo_prec
     &,precfac
      integer
     & init,icase,ii,base(3),per(3),imax
      character*2 label(3)
      data init/0/,base/4,2,1/,label/'p1','p2','p3'/
      save init,pi2,thrs
*
      parameter(precfac=1d0)
*
      if (init.eq.0) then
        init = 1
        pi2  = dcmplx( avh_olo_pi()**2 )
        thrs = precfac*avh_olo_prec()
      endif
      mu2 = avh_olo_mu_get()**2
*
      pp(1) = p1
      pp(2) = p2
      pp(3) = p3
      ff = dabs(pp(1))
      imax = 1
      do ii=2,3
        gg = dabs(pp(ii))
        if (gg.gt.ff) then
          ff = gg
          imax = ii
        endif
      enddo
      ff = ff*thrs
      icase = 0
      do ii=1,3
      if (pp(ii).ne.0d0) then
        icase = icase + base(ii)
        gg = dabs(pp(ii))
        if (gg.lt.ff) write(6,*) 'WARNING from avh_olo_c0: '
     &    ,'|',label(ii),'/',label(imax),'| =',gg/ff*thrs
      endif
      enddo
      call avh_olo_c0per(icase,per)
*
      if     (icase.eq.0) then
* 0 masses non-zero
       write(6,*) 'ERROR in avh_olo_c0: all external masses equal zero'
        stop
      elseif (icase.eq.1) then
* 1 mass non-zero
        log3 = avh_olo_log(-pp(per(3))/mu2,-1 ,19)
        valc0(2) = dcmplx( 1d0/pp(per(3)) )
        valc0(1) = -log3/pp(per(3))
        valc0(0) = ( log3**2/2 - pi2/12 )/pp(per(3))
      elseif (icase.eq.2) then
* 2 masses non-zero
        log2 = avh_olo_log(-pp(per(2))/mu2,-1 ,20)
        log3 = avh_olo_log(-pp(per(3))/mu2,-1 ,21)
        valc0(2) = dcmplx(0d0)
        valc0(1) = -avh_olo_log3( pp(per(3)) , 1 , pp(per(2)) , 1 )
        valc0(0) = -valc0(1)*(log3+log2)/2
      elseif (icase.eq.3) then
* 3 masses non-zero
        valc0(2) = dcmplx(0d0)
        valc0(1) = dcmplx(0d0)
        call avh_olo_c03m(valc0(0),p1,p2,p3 ,1)
      endif
      end
*
      subroutine avh_olo_c0per(icase,per)
*  ********************************************************************
*  * Go through all possibilities of zero (0) and non-zero (1) masses
*  *
*  *   mass: 123    mass: 123    mass: 123
*  * icase=1 001  icase=3 011  icase=0 000 icase->0
*  * icase=2 010  icase=6 110  icase=7 111 icase->3 
*  * icase=4 100  icase=5 101
*  *   icase->1     icase->2
*  ********************************************************************
      implicit none
      integer icase,per(3)
     &,permtable(3,0:7),casetable(0:7),ii
      data permtable/
     & 1,2,3 ! 0, 0 masses non-zero, no permutation
     &,1,2,3 ! 1, 1 mass non-zero,   no permutation
     &,3,1,2 ! 2, 1 mass non-zero,   1 cyclic permutation
     &,1,2,3 ! 3, 2 masses non-zero, no permutation
     &,2,3,1 ! 4, 1 mass non-zero,   2 cyclic permutations
     &,2,3,1 ! 5, 2 masses non-zero, 2 cyclic permutations
     &,3,1,2 ! 6, 2 masses non-zero, 1 cyclic permutation
     &,1,2,3 ! 7, 3 masses non-zero, no permutation
     &/             ! 0, 1, 2, 3, 4, 5, 6, 7
      data casetable/ 0, 1, 1, 2, 1, 2, 2, 3/
      do ii=1,3
        per(ii) = permtable(ii,icase)
      enddo
      icase = casetable(icase)
      end
*
      subroutine avh_olo_c03m(valc0,p1,p2,p3 ,sgneps)
*  ********************************************************************
*  * sgneps = sign causal epsilon
*  ********************************************************************
      implicit none
      double complex valc0
     &,clam,logx1x2,cx,one,y1,y2
     &,avh_olo_log,avh_olo_log3,avh_olo_log3c,avh_olo_li2
     &,avh_olo_li2c,avh_olo_li23,avh_olo_li23c
      double precision p1,p2,p3
     &,pp(3),rlam,ap(3),x1,x2,hh,x1x2,ax2,pi,avh_olo_pi,avh_olo_log4
     &,x1o,x2o,xx(4)
      integer sgneps
     &,per(3),mag(3),sgn(3),ix(2,2),per1,init,sgnal
      logical l_lt_0
      data init/0/
      save init,pi
*
      if (init.eq.0) then
        init = 1
        pi = avh_olo_pi()
      endif
      pp(1) = p1
      pp(2) = p2
      pp(3) = p3
      ap(1) = dabs(pp(1))
      ap(2) = dabs(pp(2))
      ap(3) = dabs(pp(3))
      per(1) = 1
      per(2) = 2
      per(3) = 3
      sgn(1) = 1
      sgn(2) = 1
      sgn(3) = 1
      if (pp(1).lt.0d0) sgn(1) = -1
      if (pp(2).lt.0d0) sgn(2) = -1
      if (pp(3).lt.0d0) sgn(3) = -1
*
      if (sgn(1).eq.sgn(2).and.sgn(2).eq.sgn(3)) then
        if (ap(per(2)).lt.ap(per(1))) then
          per1   = per(1)
          per(1) = per(2)
          per(2) = per1
        endif
        if (ap(per(3)).lt.ap(per(1))) then
          per1   = per(1)
          per(1) = per(3)
          per(3) = per1
        endif
      else
        if     (sgn(2).ne.sgn(3).and.sgn(2).ne.sgn(1)) then
          per(1) = 2
          per(2) = 3
          per(3) = 1
        elseif (sgn(3).ne.sgn(1).and.sgn(3).ne.sgn(2)) then
          per(1) = 3
          per(2) = 1
          per(3) = 2
        endif
      endif
*
      call avh_olo_kallen(l_lt_0,rlam,mag ,pp)
*
      if (l_lt_0) then
        hh = pp(mag(1)) + pp(mag(2))
        hh = hh + pp(mag(3))
        if (hh.gt.0d0) then
          clam = dcmplx( 0d0, -sgneps*rlam )
        else
          clam = dcmplx( 0d0,  sgneps*rlam )
        endif
        one = dcmplx(1d0)
        cx  = (dcmplx( pp(per(1))+pp(per(2))-pp(per(3)) ) 
     &                                            + clam)/pp(per(1))/2
        ax2 = cdabs(cx)**2
        y2  = cx/ax2
        y1  = dconjg(y2) 
        valc0 = 2*avh_olo_li23c( y2 , y1 )/ax2
        valc0 = valc0
     &     + dlog(ax2)*dcmplx( avh_olo_log4(cx-one)-avh_olo_log4(cx) )
        valc0 = valc0/pp(per(1))
      else
        call avh_olo_c03m_h(xx,ix ,rlam,pp,per,mag)
        if ((0d0.le.xx(ix(1,1)).and.xx(ix(1,1)).le.1d0)
     &  .or.(0d0.le.xx(ix(1,2)).and.xx(ix(1,2)).le.1d0)) then
          x1 = xx(ix(2,1))
          x2 = xx(ix(2,2))
          x1o = -xx(ix(1,2))
          x2o = -xx(ix(1,1))
        else
          x1 = xx(ix(1,1))
          x2 = xx(ix(1,2))
          x1o = -xx(ix(2,2))
          x2o = -xx(ix(2,1))
        endif
        valc0 = 2*avh_olo_li23( 1d0/x1 , 1d0/x2 ,0)/(x1*x2)
        logx1x2 = avh_olo_log( x1*x2 , sgneps*sgn(per(1)) ,23)
        valc0 = valc0 + logx1x2*( avh_olo_log3(x1o,1 ,x2o,1)
     &                           -avh_olo_log3(x1 ,1 ,x2 ,1) )
        valc0 = valc0/pp(per(1))
      endif
      end

      subroutine avh_olo_kallen(l_lt_0,lam,mag ,pp)
*  ********************************************************************
*  * Calculate "tmp = p1^2 + p2^2 + p3^2 - 2*(p1*p2 + p2*p3 + p3*p1)".
*  * If "tmp<0", then "l_lt_0=.true." and "lam=dsqrt(-tmp)".
*  * If "tmp>=0", then "l_lt_0=.false." and "lam=dsqrt(tmp)".
*  * The integer array "mag" is such that
*  *     abs(pp(mag(1))) >= abs(pp(mag(2))) >= abs(pp(mag(3)))
*  ********************************************************************
      implicit none
      logical l_lt_0
      integer mag(3) ,ii
      double precision lam,pp(3) ,ap(3)
      ap(1) = dabs(pp(1))
      ap(2) = dabs(pp(2))
      ap(3) = dabs(pp(3))
      mag(1) = 1
      mag(2) = 2
      mag(3) = 3
      if (ap(mag(3)).gt.ap(mag(2))) then
        ii     = mag(3)
        mag(3) = mag(2)
        mag(2) = ii
      endif
      if (ap(mag(2)).gt.ap(mag(1))) then
        ii     = mag(2)
        mag(2) = mag(1)
        mag(1) = ii
        if (ap(mag(3)).gt.ap(mag(2))) then
          ii     = mag(3)
          mag(3) = mag(2)
          mag(2) = ii
        endif
      endif
*
      lam = pp(mag(2)) + pp(mag(3))
      lam = ( pp(mag(1)) - lam )**2
      lam = lam - 4d0*pp(mag(2))*pp(mag(3))
*
      l_lt_0 = (lam.lt.0d0)
      if (l_lt_0) then
        lam = dsqrt(-lam)
      else
        lam = dsqrt(lam)
      endif
      end


      subroutine avh_olo_c03m_h(xx,ix ,lam,pp,per,mag)
*  ********************************************************************
*  * We denote pj = pp(per(j))  
*  * and  lam = dsqrt( p1^2 + p2^2 + p3^2 - 2*(p1*p2 + p2*p3 + p3*p1) )
*  * Calculate the solutions to the equations
*  *   1:  x*p3 + (1-x)*p2 - x*(1-x)*p1 = 0
*  *   2:  x*p2 + (1-x)*p3 - x*(1-x)*p1 = 0
*  * given by
*  *   xx(ix(1,1)) = (p1 + p2 - p3 + lam)/2/p1
*  *   xx(ix(1,2)) = (p1 + p2 - p3 - lam)/2/p1
*  *   xx(ix(2,1)) = (p1 + p3 - p2 + lam)/2/p1
*  *   xx(ix(2,2)) = (p1 + p3 - p2 - lam)/2/p1
*  * for values of p1,p2,p3 such that lam is real.
*  * Notice that
*  *   xx(ix(1,1)) - 1 = -xx(ix(2,2))
*  *   xx(ix(1,2)) - 1 = -xx(ix(2,1))
*  *   xx(ix(2,1)) - 1 = -xx(ix(1,2))
*  *   xx(ix(2,2)) - 1 = -xx(ix(1,1))
*  ********************************************************************
      implicit none
      integer ix(2,2),per(3),mag(3)
     &,init,ip(2,3),ii,jj,iter,niter
      double precision lam,xx(4),pp(3)
     &,ap(3),ff,gg,hh,g2,g4,yy,tt,a1,a2,b1,b2,xo
     &,thrs,thrs1,avh_olo_prec
      save init,thrs1
*
      parameter(thrs=1d-2)
      parameter(niter=2)
*
      if (init.eq.0) then
        init = 1
        thrs1 = 1d-4
        if (avh_olo_prec().lt.1d-30) thrs1 = 1d-6 ! quadruple precision
      endif
*
      ix(1,1) = 1
      ix(1,2) = 2
      ix(2,1) = 3
      ix(2,2) = 4
      ip(1,2) = per(2)
      ip(1,3) = per(3)
      ip(2,2) = per(3)
      ip(2,3) = per(2)
*
* The case that |p1| is small compared to |p2| and |p3|
      if (mag(3).eq.per(1)) then
        ff = dabs(pp(mag(3))/pp(mag(1)))
        if (ff.lt.thrs1) then
          ff = pp(per(2))*pp(per(3))
          gg = pp(per(2))+pp(per(3))
          hh = dabs( pp(per(2))-pp(per(3)) )
          yy = pp(per(1))/hh**2
          tt = 0d0
          g2 = gg*gg
          g4 = g2*g2
c          tt = tt*yy + (g4+10*ff*(g2+ff))*gg
          tt = tt*yy + (g4+2*ff*(3*g2+ff))
          tt = tt*yy + (g2+3*ff)*gg
          tt = tt*yy + (g2+ff)
          tt = tt*yy + gg
          tt = tt*yy + 1d0
          tt = tt*yy*ff/hh
          do ii=1,2
            hh = pp(ip(ii,2)) - pp(ip(ii,3))
            if (hh.gt.0) then
              a1 = hh/pp(per(1))
              b1 = -pp(ip(ii,3))/hh
              a2 = 0d0
              b2 = pp(ip(ii,2))/hh
            else
              a1 = 0d0
              b1 = pp(ip(ii,2))/hh
              a2 = hh/pp(per(1))
              b2 = -pp(ip(ii,3))/hh
              tt = -tt
            endif
            xx(ix(ii,1)) = a1 + b1 + tt
            xx(ix(ii,2)) = a2 + b2 - tt
          enddo
          return
        endif
      endif
*
* The general case
      hh = 2d0*pp(per(1))
      do ii=1,2
        ff = pp(per(1)) + pp(ip(ii,2)) - pp(ip(ii,3))
        xx(ix(ii,1)) = ( ff + lam )/hh
        xx(ix(ii,2)) = ( ff - lam )/hh
      enddo
*
* If an xx is close to zero, iterate Newton-Raphson a few times
      do ii=1,2
      do jj=1,2
      if (dabs(xx(ix(ii,jj))).lt.thrs) then
        do iter=1,niter
          xo = xx(ix(ii,jj))
          hh = pp(ip(ii,3)) - pp(ip(ii,2)) - pp(per(1))
          ff = pp(ip(ii,2)) + xo*hh  + xo*xo*pp(per(1))
          gg = hh + 2d0*xo*pp(per(1))
          xx(ix(ii,jj)) = xo - ff/gg
        enddo
      endif
      enddo
      enddo
*
      end

      
      subroutine avh_olo_b0(valb0,p1)
*  ********************************************************************
*  *
*  *            C   /   d^(Dim)q
*  * valb0 = ------ | ------------ 
*  *         i*pi^2 / q^2 (q+k1)^2
*  *
*  * with  Dim = 4-2*eps
*  *         C = pi^eps * mu^(2*eps) * exp(gamma_Euler*eps)
*  *
*  * input:  p1 = k1^2
*  * output: valc0(0) = eps^0   -coefficient
*  *         valc0(1) = eps^(-1)-coefficient
*  *         valc0(2) = eps^(-2)-coefficient
*  *
*  * The input value (p1) should be real.
*  * If this number is IDENTICALLY 0d0, the IR-singular case is
*  * returned.
*  ********************************************************************
      implicit none
      double complex valb0(0:2)
     &,avh_olo_log
      double precision p1
     &,arg,mu2,thrs,avh_olo_mu_get,avh_olo_prec,precfac
      integer
     & init
      data init/0/
      save init,thrs,mu2
*
      parameter(precfac=1d1)
*
      if (init.eq.0) then
        init = 1
        mu2  = avh_olo_mu_get()**2
        thrs = precfac*avh_olo_prec()
      endif
      arg = -p1/mu2
      if (p1.eq.0d0) then
        valb0(2) = dcmplx(0d0)
        valb0(1) = dcmplx(0d0)
        valb0(0) = dcmplx(0d0)
      else
        if (dabs(arg).lt.thrs) write(6,*) 'WARNING from avh_olo_b0: '
     &    ,'|p1/mu2| =', dabs(arg)
        valb0(2) = dcmplx(0d0)
        valb0(1) = dcmplx(1d0)
        valb0(0) = dcmplx(2d0) - avh_olo_log(arg,-1 ,24)
      endif
      end
      
      function avh_olo_li23c(x1,x2)
*  ********************************************************************
*  * avh_olo_li23c = ( avh_olo_li2c(x1) - avh_olo_li2c(x2) )/(x1-x2)
*  * where
*  *                           /1    ln(1-xx*t)
*  *      avh_olo_li2c(xx) = - |  dt ----------
*  *                           /0        t
*  * uses the fortran cdlog-function directly
*  ********************************************************************
      implicit none
      double complex avh_olo_li23c ,x1,x2
     &,delta,avh_olo_li2c
      logical
     & dont_use
*
      delta = x1-x2
      call avh_olo_li23ca(dont_use,avh_olo_li23c,x2,delta)
      if (dont_use) then
        avh_olo_li23c = ( avh_olo_li2c(x1) - avh_olo_li2c(x2) )/delta
      endif
      end
*
      subroutine avh_olo_li23ca(dont_use,ff,xx,delta)
*  ********************************************************************
*  *      ff = ( avh_olo_li2c(xx+delta) - avh_olo_li2c(xx) )/delta
*  * where
*  *                           /1    ln(1-xx*t)
*  *      avh_olo_li2c(xx) = - |  dt ----------
*  *                           /0        t
*  * Uses Taylor expansion of  nn  terms around  xx.
*  * If a precision of  nn*prec  is not reached, then  dont_use=true
*  ********************************************************************
      implicit none
      double complex ff,xx,delta
     &,fi,fo,gg,hh,one,avh_olo_log3c
      double precision
     & prec,aa,bb,avh_olo_prec
      integer
     & init,nn,ii
      logical dont_use
      data init/0/
      save init,prec
*
      parameter(nn=4)
*
      if (init.eq.0) then
        init = 1
        prec = avh_olo_prec()
      endif
      one = dcmplx(1d0)
*
      fi = avh_olo_log3c(one-xx,one) ! -log(1-xx)/xx
      gg = one/(one-xx)
      hh = gg
      ff = fi
      fo = fi
      fi = (delta/2)*( hh - fo )/xx
      ff = ff + fi
      fo = fi
      dont_use = .false.
      do ii=3,nn
        hh = (delta/(ii-1))*(ii-2)*gg*hh
        fi = (delta/ii)*( hh - (ii-1)*fo )/xx
        aa = cdabs(fi)/cdabs(ff)
        bb = ii*prec
        if (aa.le.bb) goto 100
        ff = ff + fi
        fo = fi
      enddo
      dont_use = .true.
  100 continue
      end
      
      function avh_olo_li23(x1,x2,sgn)
*  ********************************************************************
*  *    avh_olo_li23
*  *    = ( avh_olo_li2(x1,sgn) - avh_olo_li2(x2,sgn) )/(x1-x2)
*  * where
*  *                              /1    ln(1-zz*t)
*  *      avh_olo_li2(xx,sgn) = - |  dt ----------
*  *                              /0        t
*  * with  
*  *            zz = xx + i*sgn*epsilon,  epsilon -> 0
*  ********************************************************************
      implicit none
      double complex avh_olo_li23
     &,avh_olo_li2
      double precision x1,x2
     &,delta
      integer sgn
      logical
     & dont_use
*
      delta = x1-x2
      call avh_olo_li23a(dont_use,avh_olo_li23,x2,delta,sgn)
      if (dont_use) then
        avh_olo_li23
     &  = ( avh_olo_li2(x1,sgn) - avh_olo_li2(x2,sgn) )/delta
      endif
      end
*
      subroutine avh_olo_li23a(dont_use,ff,xx,delta,sgn)
*  ********************************************************************
*  * ff = ( avh_olo_li2(xx+delta,sgn) - avh_olo_li2(xx,sgn) )/delta
*  * where
*  *                              /1    ln(1-zz*t)
*  *      avh_olo_li2(xx,sgn) = - |  dt ----------
*  *                              /0        t
*  * with  
*  *            zz = xx + i*sgn*epsilon,  epsilon -> 0
*  * Uses Taylor expansion of  nn  terms around  xx.
*  * If a precision of  nn*prec  is not reached, then  dont_use=true
*  ********************************************************************
      implicit none
      double complex ff
     &,fi,fo,hh,avh_olo_log3
      double precision xx,delta
     &,prec,gg,aa,bb,avh_olo_prec
      integer sgn
     &,init,nn,ii
      logical dont_use
      data init/0/
      save init,prec
*
      parameter(nn=4)
*
      if (init.eq.0) then
        init = 1
        prec = avh_olo_prec()
      endif
*
      if (xx.eq.1d0) then
        dont_use = .true.
        return
      endif
*
      fi = avh_olo_log3(1d0-xx,-sgn, 1d0,1) ! -log(1-xx)/xx
      gg = 1d0/(1d0-xx)
      hh = dcmplx(gg)
      ff = fi
      fo = fi
      fi = (delta/2)*( hh - fo )/xx
      ff = ff + fi
      fo = fi
      dont_use = .false.
      do ii=3,nn
        hh = (delta/(ii-1))*(ii-2)*gg*hh
        fi = (delta/ii)*( hh - (ii-1)*fo )/xx
        aa = cdabs(fi)/cdabs(ff)
        bb = ii*prec
        if (aa.le.bb) goto 100
        ff = ff + fi
        fo = fi
      enddo
      dont_use = .true.
  100 continue
      end
      
      function avh_olo_li2c(xx)
*  ********************************************************************
*  *                   /1    ln(1-xx*t)
*  *  avh_olo_li2c = - |  dt ----------
*  *                   /0        t
*  * uses the fortran cdlog-function directly
*  * The number of terms nn in the expansion should be set by the user.
*  ********************************************************************
      implicit none
      double complex avh_olo_li2c,xx
     &,lix,yy,zz,z2
      double precision
     & bb(20),pi2o6,avh_olo_pi,avh_olo_bern
      integer
     & init,nn,ii
      logical
     & x_gt_1 , y_gt_h
      data init/0/
      save init,bb,pi2o6
*
      parameter(nn=18)
*
      if (init.eq.0) then
        init = 1
        pi2o6  = avh_olo_pi()**2/6d0
        do ii=1,nn
          bb(ii) = avh_olo_bern(ii)
        enddo
      endif
*     
      if     (xx.eq.dcmplx(0d0)) then
        lix = dcmplx(0d0)
      elseif (xx.eq.dcmplx(1d0)) then
        lix = dcmplx(pi2o6)
      else
        x_gt_1 = (cdabs(xx).gt.1d0)
        if (x_gt_1) then
          yy = dcmplx(1d0)/xx
        else
          yy = xx
        endif
*
        y_gt_h = (dreal(yy).gt.0.5d0)
        if (y_gt_h) yy = dcmplx(1d0)-yy
*
        zz  = -cdlog(dcmplx(1d0)-yy)
        z2  = zz*zz
        lix = bb(nn)
        do ii=nn,4,-2
          lix = bb(ii-2) + lix*z2/(ii*(ii+1))
        enddo
        lix = bb(1) + lix*zz/3
        lix = zz + lix*z2/2
*
        if (y_gt_h) lix = -lix + dcmplx(pi2o6) + cdlog(yy)*zz
        if (x_gt_1) lix = -lix - dcmplx(pi2o6) - cdlog(-xx)**2/2
      endif
      avh_olo_li2c = lix
      end

      function avh_olo_li2(xx,sgn)
*  ********************************************************************
*  *                  /1    ln(1-zz*t)
*  *  avh_olo_li2 = - |  dt ----------
*  *                  /0        t
*  * where  zz = xx + i*sgn*epsilon,  epsilon -> 0
*  * The number of terms nn in the expansion shoud be set by the user.
*  ********************************************************************
      implicit none
      double complex avh_olo_li2
     &,zz,z2,avh_olo_log
      double precision xx
     &,bb(30),pi2o6,yy,avh_olo_pi,avh_olo_bern
      integer sgn
     &,init,nn,ii,sy
      logical
     & x_gt_1 , y_gt_h
      data init/0/
      save init,bb,pi2o6
*
      parameter(nn=18)
*
      if (init.eq.0) then
        init = 1
        pi2o6  = avh_olo_pi()**2/6d0
        do ii=1,nn
          bb(ii) = avh_olo_bern(ii)
        enddo
      endif
*     
      if     (xx.eq.0d0) then
        avh_olo_li2 = dcmplx(0d0)
      elseif (xx.eq.1d0) then
        avh_olo_li2 = dcmplx(pi2o6)
      else
        x_gt_1 = (dabs(xx).gt.1d0)
        if (x_gt_1) then
          yy = 1d0/xx
          sy = -sgn
        else
          yy = xx
          sy = sgn
        endif
*
        y_gt_h = (yy.gt.0.5d0)
        if (y_gt_h) then
          yy = 1d0-yy
          sy = -sy
        endif
*
        zz = -avh_olo_log(1d0-yy,-sy ,25)
        z2 = zz*zz
        avh_olo_li2 = bb(nn)
        do ii=nn,4,-2
          avh_olo_li2 = bb(ii-2) + avh_olo_li2*z2/(ii*(ii+1))
        enddo
        avh_olo_li2 = bb(1) + avh_olo_li2*zz/3
        avh_olo_li2 = zz + avh_olo_li2*z2/2
*
        if (y_gt_h) avh_olo_li2 = -avh_olo_li2 + dcmplx(pi2o6)
     &                           + avh_olo_log(yy,sy ,26)*zz
        if (x_gt_1) avh_olo_li2 = -avh_olo_li2 - dcmplx(pi2o6) 
     &                           - avh_olo_log(-xx,-sgn ,27)**2/2
      endif
      end

      function avh_olo_log4(xx)
*  ********************************************************************
*  * avh_olo_log4 = log( xx/conjg(xx) )/( xx-conjg(xx) )
*  ********************************************************************
      implicit none
      double complex xx
      double precision avh_olo_log4
     &,xr,xi,tg,pi,avh_olo_pi,avh_olo_atan
      integer
     & init
      data init/0/
      save init,pi
*
      if (init.eq.0) then
        init = 1
        pi = avh_olo_pi()
      endif
      xr = dreal(xx)
      xi = dimag(xx)
      if     (xr.eq.0d0) then ! take log(-1)=pi
        avh_olo_log4 = dcmplx(pi/xi/2)
      else
        tg = xi/xr
        if (xr.gt.0d0.and.dabs(tg).lt.1d0) then ! take care of small xi
          avh_olo_log4 = avh_olo_atan(tg)/xr
        else
          avh_olo_log4 = dimag(cdlog(xx))/xi
        endif
      endif
      end
*
      function avh_olo_atan(xx)
*  ********************************************************************
*  * avh_olo_atan = arctan(xx)/xx  
*  ********************************************************************
      implicit none
      double precision avh_olo_atan,xx
     &,thrs,temp,prec,avh_olo_prec
      integer
     & init,ii
      data init/0/
      save init,thrs
*
      if (init.eq.0) then
        init = 1
        prec = avh_olo_prec()
        thrs = 1d0
  100   temp = dabs( (datan(thrs)/thrs)/(1d0-thrs*thrs/3)  - 1d0 )
        if (temp.gt.prec) then
          thrs = thrs/2
          goto 100
        endif
      endif
      if (dabs(xx).gt.thrs) then
        avh_olo_atan = datan(xx)/xx
      else
        avh_olo_atan = 1d0 - xx*xx/3
      endif
      end
 
      function avh_olo_log3c(x1,x2)
*  ********************************************************************
*  * avh_olo_log3c = log(x1/x2)/(x1-x2)  
*  * uses the fortran cdlog-function directly
*  ********************************************************************
      implicit none
      double complex avh_olo_log3c ,x1,x2
     &,y1,y2,y1oy2,delta
      double precision
     & thrs,temp,prec,avh_olo_prec
      integer
     & init,ii
      data init/0/
      save init,thrs
*
      if (init.eq.0) then
        init = 1
        prec = avh_olo_prec()
        thrs = 1d0
  100   temp = dabs( (dlog(1d0+thrs)/thrs) / (1d0-thrs/2) - 1d0 )
        if (temp.gt.prec) then
          thrs = thrs/2
          goto 100
        endif
      endif
      y1 = x1
      y2 = x2
      if (cdabs(y1).gt.cdabs(y2)) then
        y1 = x2
        y2 = x1
      endif
      y1oy2 = y1/y2
      delta = y1oy2 - dcmplx(1d0)
      if (cdabs(delta).gt.thrs) then
        avh_olo_log3c =  cdlog(y1oy2)/(y1-y2)
      else
        avh_olo_log3c = (dcmplx(1d0)-delta/2)/y2
      endif
      end
 
      function avh_olo_log3(x1,sg1,x2,sg2)
*  ********************************************************************
*  * avh_olo_log3 = log( (x1+i*sg1*eps)/(x2+i*sg2*eps) ) / (x1-x2)  
*  * with  eps->0
*  ********************************************************************
      implicit none
      double complex avh_olo_log3
     &,avh_olo_log
      double precision x1,x2
     &,thrs,temp,delta,y1,y2,y1oy2,prec,avh_olo_prec
      integer sg1,sg2
     &,init,ii,s1,s2,sgn
      data init/0/
      save init,thrs
*
      if (init.eq.0) then
        init = 1
        prec = avh_olo_prec()
        thrs = 1d0
  100   temp = dabs( (dlog(1d0+thrs)/thrs) / (1d0-thrs/2) - 1d0 )
        if (temp.gt.prec) then
          thrs = thrs/2
          goto 100
        endif
      endif
      y1 = x1
      s1 = sg1
      y2 = x2
      s2 = sg2
      if (dabs(y1).gt.dabs(y2)) then
        y1 = x2
        s1 = sg2
        y2 = x1
        s2 = sg1
      endif
      y1oy2 = y1/y2
      delta = y1oy2 - 1d0
      if (dabs(delta).gt.thrs) then
        sgn = 1
        if ( (s1*y2-s2*y1).lt.0d0 ) sgn = -1
        avh_olo_log3 =  avh_olo_log( y1oy2 , sgn ,28 )/(y1-y2)
      else
        avh_olo_log3 = dcmplx( (1d0-delta/2)/y2 )
      endif
      end
 
      function avh_olo_log2(x1,sg1,x2,sg2)
*  ********************************************************************
*  * avh_olo_log2 = log( (x1+i*sg1*eps)/(x2+i*sg2*eps) )  with  eps->0
*  ********************************************************************
      implicit none
      double complex avh_olo_log2
     &,avh_olo_log
      double precision x1,x2
     &,y1,y2
      integer sg1,sg2
     &,s1,s2,so,sgn
*
      y1 = x1
      s1 = sg1
      y2 = x2
      s2 = sg2
      so = 1
      if (dabs(x1).gt.dabs(x2)) then
        y1 = x2
        s1 = sg2
        y2 = x1
        s2 = sg1
        so = -1
      endif
      sgn = 1
      if ( (s1*y2-s2*y1).lt.0d0 ) sgn = -1
      avh_olo_log2 =  so*avh_olo_log( y1/y2 , sgn  ,29)
      end
 
      function avh_olo_log(xx,sgn ,tag)
*  ********************************************************************
*  * avh_olo_log = log( xx + i*sgn*eps )  with  eps->0
*  ********************************************************************
      implicit none
      double complex avh_olo_log
      double precision xx
     &,pi,avh_olo_pi
      character*24 text
      integer sgn ,tag
     &,init
      data init/0/
      save init,pi
*
      if (init.eq.0) then
        init = 1
        pi = avh_olo_pi()
      endif
      if (xx.gt.0d0) then
        avh_olo_log = dcmplx( dlog(xx)  , 0d0 )
      elseif ((xx.lt.0d0).and.(sgn.ne.0)) then
        avh_olo_log = dcmplx( dlog(-xx) , sgn*pi )
      else
        write(6,*) 'ERROR in avh_olo_log at tag',tag
     &            ,' :  xx =',xx,',  sng =',sgn
      endif
      end
 
      function avh_olo_pi()
*  ********************************************************************
*  * the number  pi=3.14...
*  ********************************************************************
      implicit none
      double precision avh_olo_pi
     &,pi
      integer
     & init
      data init/0/
      save init,pi
*
      if (init.eq.0) then
        init = 1
        pi = 4*datan(1d0)
      endif
      avh_olo_pi = pi
      end
 
      function avh_olo_bern(ii)
*  ********************************************************************
*  * the first nn Bernoulli numbers
*  ********************************************************************
      implicit none
      double precision avh_olo_bern
     &,bern(30)
      integer ii
     &,init,nn,jj
      data init/0/
      save init,bern
*
      parameter(nn=30)
*
      if (init.eq.0) then
        init = 1
        do jj=3,nn-1,2
          bern(jj) = 0d0
        enddo
        bern( 1) = -1d0/2d0
        bern( 2) =  1d0/6d0
        bern( 4) = -1d0/30d0
        bern( 6) =  1d0/42d0
        bern( 8) = -1d0/30d0
        bern(10) =  5d0/66d0
        bern(12) = -691d0/2730d0
        bern(14) =  7d0/6d0
        bern(16) = -3617d0/510d0
        bern(18) =  43867d0/798d0
        bern(20) = -174611d0/330d0
        bern(22) =  854513d0/138d0
        bern(24) = -236364091d0/2730d0
        bern(26) =  8553103d0/6d0
        bern(28) = -23749461029d0/870d0
        bern(30) =  8615841276005d0/14322d0
      endif
      if (ii.le.nn) then
        avh_olo_bern = bern(ii)
      else
        write(6,*) 'ERROR in avh_olo_bern: bernoulli(',ii,
     &             ') not yet implemented'
      endif
      end
 
      function avh_olo_prec()
*  ********************************************************************
*  * the smallest number  prec  satisfying  1+prec = dexp(dlog(1+prec))
*  ********************************************************************
      implicit none
      double precision avh_olo_prec
     &,prec,xx,yy
      integer
     & init ,ii
      data init/0/
      save init,prec
*
      if (init.eq.0) then
        init = 1
        xx = 1d0
        yy = xx
        do while (xx.eq.yy)
          prec = xx
          xx = xx/2
          yy = -1d0+dexp(dlog(1d0+xx))
        enddo
        call avh_olo_hello
        write(6,*) 'MESSAGE from avh_olo_prec: '
     &            ,'precision set to',prec
      endif
      avh_olo_prec = prec
      end


      subroutine avh_olo_hello
*  ********************************************************************
*  ********************************************************************
      implicit none
      write(*,'(a36,a36)') '####################################'
     &                    ,'####################################'
      write(*,'(a36,a36)') '#                                   '
     &                    ,'                                   #'
      write(*,'(a36,a36)') '#                         You are us'
     &                    ,'ing OLO_S4                         #'
      write(*,'(a36,a36)') '#                                   '
     &                    ,'                                   #'
      write(*,'(a36,a36)') '# for the evaluation of one-loop sca'
     &                    ,'lar 2-, 3- and 4-point functions   #'
      write(*,'(a36,a36)') '# with all internal masses equal zer'
     &                    ,'o.                                 #'
      write(*,'(a36,a36)') '#                                   '
     &                    ,'                                   #'
      write(*,'(a36,a36)') '# author: Andreas van Hameren <Andre'
     &                    ,'.HamerenREMOVETHIS@ifj.edu.pl>     #'
      write(*,'(a36,a36)') '#   date: 20-11-2007                '
     &                    ,'                                   #'
      write(*,'(a36,a36)') '#                                   '
     &                    ,'                                   #'
      write(*,'(a36,a36)') '# Please cite                       '
     &                    ,'                                   #'
      write(*,'(a36,a36)') '#    A. van Hameren, J. Vollinga, S.'
     &                    ,' Weinzierl,                        #'
      write(*,'(a36,a36)') '#    Eur.Phys.J.C41:361-375,2005 (he'
     &                    ,'p-ph/0502165)                      #'
      write(*,'(a36,a36)') '# in publications with results obtai'
     &                    ,'ned with the help of this program. #'
      write(*,'(a36,a36)') '#                                   '
     &                    ,'                                   #'
      write(*,'(a36,a36)') '####################################'
     &                    ,'####################################'
      end
