#include <getopt.h>
#include <stdlib.h>
#include <iostream>
#include <math.h>
#include <stdio.h>
#include <vector>
#include <algorithm>

using namespace std;

int seed = 0;	
int nParticipants=0,nMeetings=0,nTimeslots=0,nLocations=-1;
//char* output = NULL;
FILE *fout = stdout;

vector<pair<int,int> > meetings;
vector<int> meetsPerParticipants;

vector<vector<int> > forbidden;
vector<pair<int,int> > fixedMeet;
vector<vector<int> > precedences;

bool onlyList = false;


/************************************************************************/


void printUsage(char* app){
	cerr << "Usage: " << app << " [options] -P <p> -M <m> -T <t> -L <l>" << endl;
	cerr << "  Options:" << endl;
	cerr << "     -P <int>    : number of participants" << endl;
	cerr << "     -M <int>    : number of meetings" << endl;
	cerr << "     -T <float>  : number of timeslots" << endl;
	cerr << "     -L <float>  : number of locations" << endl;
	cerr << "     -s <int>    : seed (default=0)" << endl;
	cerr << "     -o <string> : output file (default=stdout)" << endl;
	cerr << "     -l          : print only list of meetings (default=false)" << endl;
	cerr << "  Constraints:" << endl;
	cerr << "     -> p > 1" << endl;
	cerr << "     -> {m, t, l} > 0" << endl;
	cerr << "     -> m <= t * l" << endl;
	cerr << "     -> m <= (p \\choose 2)" << endl;
	cerr << "     -> m <= (p * t) / 2" << endl;
	cerr << "  Checks:" << endl;
	cerr << "     -> No repeated meetings" << endl;
	cerr << "     -> No participant with more meetings than #timeslots" << endl;
	exit(-1);
}

void parseArgs(int argc, char **argv){
	int opt;
	while((opt=getopt(argc, argv, "P:M:T:L:s:o:?hl")) != -1){
		switch(opt){
			case 'P':
				nParticipants=atoi(optarg);
				break;
			case 'M':
				nMeetings=atoi(optarg);
				break;
			case 'T':
				nTimeslots=atoi(optarg);
				break;
			case 'L':
				nLocations=atoi(optarg);
				break;
			case 's':
				seed = atoi(optarg);
				srand(seed); // Initialize seed
				break;
			case 'o':
		    	if ((fout = fopen(optarg,"w"))==NULL){
					cerr << "ERROR Unable to open file <" << optarg << ">" << endl;
					exit(-1);
		      	}
				break;
			case 'l':
				onlyList = true;
				break;
			case 'h':
			case '?':
				printUsage(argv[0]);
				break;
			default:
				cerr << "ERROR: Incorrect argument: " << optarg << endl;
				printUsage(argv[0]);
		}
	}
	
	
	
	// Check constraints:
	
	if(nParticipants<2){
		cerr << "ERROR: There is not at least two participants (#p=" 
			<< nParticipants << ")" << endl << endl;
		printUsage(argv[0]);
	}
	
	if(nMeetings<1){
		cerr << "ERROR: There is not at least one meeting (#m=" 
			<< nMeetings << ")" << endl << endl;
		printUsage(argv[0]);
	}
	
	if(nTimeslots<1){
		cerr << "ERROR: There is not at least one timeslot (#t=" 
			<< nTimeslots << ")" << endl << endl;
		printUsage(argv[0]);
	}
	
	if(nLocations<1){
		cerr << "ERROR: There is not at least one location (#l="
			<< nLocations << ")" << endl << endl;
		printUsage(argv[0]);
	}
	
	
	
	if(nMeetings > nTimeslots * nLocations){
		cerr << "ERROR: the number of meetings (#m=" << nMeetings 
			<< ") exceeds the event capacity: #t * #l = " << nTimeslots 
			<< " * " << nLocations << " = " << nTimeslots * nLocations << endl << endl;
		printUsage(argv[0]);
	}
	
	if(nMeetings > (nParticipants * (nParticipants - 1)) / 2){
		cerr << "ERROR: the number of meetings (#m=" << nMeetings 
			<< ") exceeds the participants combinations: #p \\choose 2 = " 
			<< nParticipants * (nParticipants - 1) / 2 << endl << endl;
		printUsage(argv[0]);
	}
	
	if(nMeetings > (nParticipants * nTimeslots) / 2){
		cerr << "ERROR: the number of meetings (#m=" << nMeetings 
			<< ") exceeds the meetings per partipant wrt #timeslots: ()#p * #t)/2 = " 
			<< (nParticipants * nTimeslots) / 2 << endl << endl;
		printUsage(argv[0]);
	}
}


/************************************************************************/


/*
 * Checks whether the number of meetings
 * of any participant exceed the number
 * of timeslots
 */
bool participantExceedTimeslots(){
	bool hasMore = false;
	for(int i=0; i<meetsPerParticipants.size(); i++){
		if(meetsPerParticipants[i] > nTimeslots){
			hasMore = true;
			break;
		}
	}
	return hasMore;
}

/*
 * Check whether a meeting between participants
 * p1 and p2 (parameters) already exists
 */
bool existingMeeting(int p1, int p2){
	bool exists = false;
	for(int i=0; i<meetings.size(); i++){
		if ( (meetings[i].first == p1 && meetings[i].second == p2)
			|| (meetings[i].first == p2 && meetings[i].second == p1) ){
				exists = true;
				break;
			}
	}
	return exists;
}


/************************************************************************/


/*
 * Auxiliary functions to print the list of generated meetings
 */
bool mysort(pair<int,int> a, pair<int,int> b){
	if(a.first < b.first || (a.first == b.first && a.second < b.second))
		return true;
	else
		return false;
}

void sortAndPrint(){
	// In each meeting: First participant < Second participants
	for(int i=0; i<meetings.size(); i++){
		if(meetings[i].first > meetings[i].second){
			int k = meetings[i].first;
			meetings[i].first = meetings[i].second;
			meetings[i].second = k;
		}
	}
	
	// Sort vector of meetings
	sort(meetings.begin(), meetings.end(), mysort);
	
	// Print pairs
	for(int i=0; i<meetings.size(); i++)
		cout << meetings[i].first << " " << meetings[i].second << endl;
}




/************************************************************************/

/*
 * Print the output B2B random instance, in DZN format
 */
void printDZN(){
	fprintf(fout, "nBusiness = %d;\n", nParticipants);
	fprintf(fout, "nMeetings = %d;\n", nMeetings);
	fprintf(fout, "nTables = %d;\n", nLocations);
	fprintf(fout, "nTotalSlots = %d;\n", nTimeslots);
	fprintf(fout, "nMorningSlots = %d;\n", nTimeslots);

	// List of requested meetings
	fprintf(fout, "\nrequested = [");
	for(int i=0; i<meetings.size(); i++){
		fprintf(fout, "|%d, %d, 3, \n", meetings[i].first, meetings[i].second);
	}
	fprintf(fout, "|];\n");
	
	fprintf(fout, "\nnMeetingsBusiness = [");
	for(int i=0; i<meetsPerParticipants.size(); i++){
		fprintf(fout, "%d, ", meetsPerParticipants[i]);
	}
	fprintf(fout, "];\n");
	
	
	// Forbidden
	fprintf(fout, "\nforbidden = [");
	for(int i=0; i<forbidden.size(); i++){
		fprintf(fout, "{");
		for(int j=0; j<forbidden[i].size(); j++){
			fprintf(fout, "%d,", forbidden[i][j]);
		}
		fprintf(fout, "},\n");
	}
	fprintf(fout, "];\n");
	
	
	// Fixed
	fprintf(fout, "\nfixedMeet = [");
	for(int i=0; i<fixedMeet.size(); i++){
		fprintf(fout, "|%d,%d,\n", fixedMeet[i].first, fixedMeet[i].second);
	}
	fprintf(fout, "];\n");
	
	
	// Precedences
	fprintf(fout, "\nprecedences = [");
	for(int i=0; i<precedences.size(); i++){
		fprintf(fout, "{");
		for(int j=0; j<precedences[i].size(); j++){
			fprintf(fout, "%d,", precedences[i][j]);
		} 
		fprintf(fout, "},\n");
	}
	fprintf(fout, "];\n");
	
	
	fprintf(fout, "\n");
	fclose(fout);
}





/************************************************************************/





int main(int argc, char **argv){
	
	// Parse arguments
	parseArgs(argc, argv); 

	forbidden.resize(nParticipants);
	precedences.resize(nMeetings);
	
	
		
	// Initializa vectors
	meetings.clear();
	meetsPerParticipants.clear();
	meetsPerParticipants.resize(nParticipants, 0);
	
	// Repeat for every meetings
	for(int i=0; i<nMeetings; i++){
		
		// Choose two distinct participants
		int p1,p2;
		
		// Repeat until soft feasibility conditions passed, i.e.
		//     a) no repeated meeting
		//     b) no participant with more meetings than timeslots
		do{
			p1 = rand() % nParticipants;
			p2 = rand() % nParticipants;
		}while( p1 == p2 || existingMeeting(p1,p2) || meetsPerParticipants[p1]==nTimeslots || meetsPerParticipants[p2]==nTimeslots);
		
		// Add the meeting
		meetings.push_back(make_pair(p1+1,p2+1));
		
		// Increase the counters for p1 and p2
		meetsPerParticipants[p1]++;
		meetsPerParticipants[p2]++;
	}
	
	
	// If you want to check the generated list of meetings:
	if(onlyList){
		sortAndPrint();
	}else{
		printDZN();
	}
	
	
}