% @=============================================================================
% Copyright (c)2025 University of Granada - SPAIN
% This software is distributed under the terms of the GNU General Public License
% as published by the Free Software Foundation. Further details on the GPLv3
% license can be found at http://www.gnu.org/copyleft/gpl.html
% 
% FOR RESEARCH PURPOSES ONLY. THE SOFTWARE IS PROVIDED "AS IS," AND THE
% UNIVERSITY OF GRANADA DO NOT MAKE ANY WARRANTY, EXPRESS OR IMPLIED, INCLUDING 
% BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
% PURPOSE, NOR DO THEY ASSUME ANY LIABILITY OR RESPONSIBILITY FOR THE USE 
% OF THIS SOFTWARE.
% =============================================================================@
%
% Author: Juan Ruiz de Miras, 2025

function [normalizedHeatmap] = heatMapProbVeronese(imageSize, patchSize, overlap, score, painting)
    stride = round(patchSize * (1 - overlap));
    veroneseProbs = score(:, 2);
    numPatchesPerRow = floor((imageSize - patchSize) / stride) + 1;
    
    heatmap = zeros(imageSize, imageSize);
    counter = zeros(imageSize, imageSize);
    
    index = 1;
    for y = 1:stride:(imageSize - patchSize + 1)
        for x = 1:stride:(imageSize - patchSize + 1)
            heatmap(y:y+patchSize-1, x:x+patchSize-1) = ...
                heatmap(y:y+patchSize-1, x:x+patchSize-1) + veroneseProbs(index);
            counter(y:y+patchSize-1, x:x+patchSize-1) = ...
                counter(y:y+patchSize-1, x:x+patchSize-1) + 1;
            index = index + 1;
        end
    end
    
    % normalization and Gaussian smoothing
    coverage = (numPatchesPerRow - 1) * stride + patchSize;
    heatmap = heatmap(1:coverage, 1:coverage, :);
    counter = counter(1:coverage, 1:coverage, :);
    normalizedHeatmap = heatmap ./ counter;
    normalizedHeatmap = imgaussfilt(normalizedHeatmap, 4);

    % original image
    basePath = './dataset/test/';
    baseName = [painting '_test_'];
    fileName = fullfile(basePath, [baseName 'color.bmp']);
    originalImage = imresize(imread(fileName), [imageSize imageSize]);
    originalImage = originalImage(1:coverage, 1:coverage, :);

    % show overlay: original painting + heatmap
    figure;
    subplot(1, 2, 1);
    imshow(originalImage);
    hold on;

    alphaMap = normalizedHeatmap; 
    alphaMap = alphaMap - min(alphaMap(:));
    alphaMap = alphaMap / max(alphaMap(:));
    alphaMap = 1 - alphaMap;

    h = imagesc(normalizedHeatmap);
    colormap hot;
    clim([0 1]);
    set(h, 'AlphaData', alphaMap);

    averageProb = mean(veroneseProbs) * 100;
    title(sprintf('Veronese probability: %.1f%%', averageProb), 'FontSize', 20);
    axis image off;

    % show heatmap
    subplot(1, 2, 2);
    imagesc(normalizedHeatmap);
    colormap hot;
    clim([0 1]);
    cb = colorbar;
    cb.FontSize = 16;
    cb.Ticks = [0.0 0.2 0.4 0.6 0.8 1.0];
    ylabel(cb, 'Veronese Probability', 'FontSize', 18);
    axis image off;
end
