% @=============================================================================
% Copyright (c)2025 University of Granada - SPAIN
% This software is distributed under the terms of the GNU General Public License
% as published by the Free Software Foundation. Further details on the GPLv3
% license can be found at http://www.gnu.org/copyleft/gpl.html
% 
% FOR RESEARCH PURPOSES ONLY. THE SOFTWARE IS PROVIDED "AS IS," AND THE
% UNIVERSITY OF GRANADA DO NOT MAKE ANY WARRANTY, EXPRESS OR IMPLIED, INCLUDING 
% BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
% PURPOSE, NOR DO THEY ASSUME ANY LIABILITY OR RESPONSIBILITY FOR THE USE 
% OF THIS SOFTWARE.
% =============================================================================@
%
% Author: Juan Ruiz de Miras, 2025

clear all;

% CPU and GPU seed for random calculations and reproducibility
seed = 123456; 
rng(seed); 
gpurng(seed); 
deep.gpu.deterministicAlgorithms(true);

% parameters for sliding-windows
patchSize = [64, 64];
overlap = 0.65;

class = {'veronese', 'non-veronese'};
veronesePaintings = {'01', '02', '03', '04'};
nonVeronesePaintings = {'05', '06'};

% sliding-window processing
[veroneseColorPatches, veroneseGrayscalePatches, veroneseEdgePatches] = extractPatches(patchSize, overlap, class{1}, veronesePaintings);
[nonVeroneseColorPatches, nonVeroneseGrayscalePatches, nonVeroneseEdgePatches] = extractPatches(patchSize, overlap, class{2}, nonVeronesePaintings);
[X, Y] = prepareDataCNN(veroneseColorPatches, veroneseGrayscalePatches, veroneseEdgePatches, ...
                        nonVeroneseColorPatches, nonVeroneseGrayscalePatches, nonVeroneseEdgePatches, ...
                        patchSize);

% defining CNN architecture
layers = [
    imageInputLayer([patchSize(1) patchSize(2) 5], 'Normalization', 'none')

    convolution2dLayer(3, 16, 'Padding', 'same')
    batchNormalizationLayer
    reluLayer
    maxPooling2dLayer(2, 'Stride', 2)

    convolution2dLayer(3, 32, 'Padding', 'same')
    batchNormalizationLayer
    reluLayer
    maxPooling2dLayer(2, 'Stride', 2)

    dropoutLayer(0.5)

    fullyConnectedLayer(2)
    softmaxLayer
];

% Initialize vectors for predictions and scores
y_pred = zeros(size(Y));
scores = zeros(length(Y), 1); 

% Cross-validation training on patches from all paintings except for the painting used for validation
totalPatches = length(y_pred);
numPaintings = length(veronesePaintings) + length(nonVeronesePaintings);
patchesPerPainting = totalPatches / numPaintings; 
for i = 1:numPaintings
    disp(['Training and validating partition ' num2str(i) '/' num2str(numPaintings) '...']);

    testIdx = zeros(1, totalPatches);
    iniPatch = (i - 1) * patchesPerPainting + 1;
    endPatch = i * patchesPerPainting;
    testIdx(iniPatch:endPatch) = 1;
    testIdx = logical(testIdx);
    trainIdx = testIdx == 0; % train patches are the patches that are not test patches
    
    XTrain = X(:, :, :, trainIdx);
    YTrain = Y(trainIdx);
    XVal = X(:, :, :, testIdx);
    YVal = Y(testIdx);

    % training parameters
    miniBatchSize = 128; 
    numTrainImages = size(XTrain,4);
    iterationsPerEpoch = ceil(numTrainImages / miniBatchSize);
    
    options = trainingOptions('adam', ...
                              'MaxEpochs', 100, ... 
                              'MiniBatchSize', miniBatchSize, ...
                              'Shuffle', 'every-epoch', ... 
                              'L2Regularization', 0.0005, ... % regularization to avoid overfitting from overlapping patches
                              'ValidationData', {XVal, YVal'}, ...
                              'ValidationFrequency', iterationsPerEpoch, ...
                              'ValidationPatience', 8, ... % early stopping   
                              'Metrics','accuracy', ...
                              'OutputNetwork', 'best-validation', ... 
                              'Verbose', true, ...
                              'Plots', 'none');

    % CNN training
    net = trainnet(XTrain, YTrain', layers, 'crossentropy', options);

    % CNN validation for the current fold
    [score_fold] = predict(net, XVal);
    [y_pred_fold, ~] = scores2label(score_fold, {'0', '1'});
    y_pred(testIdx) = str2double(string(y_pred_fold));
    scores(testIdx) = score_fold(:,2); 
end

% confusion matrix
confMat = confusionmat(str2double(string(Y)), y_pred);
TP = confMat(2,2);
TN = confMat(1,1);
FP = confMat(1,2);
FN = confMat(2,1);

% performance metrics
fprintf('Performance metrics for %d-fold cross-validation\n', numPaintings);
accuracy = (TP + TN) / sum(confMat(:));
precision = TP / (TP + FP);
sensitivity = TP / (TP + FN);
specificity = TN / (TN + FP);
f1_score = 2 * (precision * sensitivity) / (precision + sensitivity);

Gmean = sqrt(sensitivity * specificity);
po = accuracy;
pe = ((TP + FP)*(TP + FN) + (FN + TN)*(FP + TN)) / sum(confMat(:))^2;
kappa = (po - pe) / (1 - pe);

fprintf('Accuracy     : %.2f%%\n', accuracy * 100);
fprintf('Precision    : %.2f%%\n', precision * 100);
fprintf('F1-Score     : %.2f%%\n', f1_score * 100);
fprintf('Sensitivity  : %.2f%%\n', sensitivity * 100);
fprintf('Specificity  : %.2f%%\n', specificity * 100);
fprintf('G-mean       : %.2f%%\n', Gmean * 100);
fprintf('Kappa        : %.2f \n', kappa);

% ROC curve
[Xroc, Yroc, T, AUC] = perfcurve(Y, scores, '1');
figure;
plot(Xroc, Yroc, 'b-', 'LineWidth', 2);
hold on;
plot([0 1], [0 1], 'r--');
xlabel('False Positive Rate (1 - Specificity)'); 
ylabel('True Positive Rate (Sensitivity)');
title(['ROC Curve (AUC = ', num2str(AUC, '%.2f'), ')']); 
legend('CNN Model', 'Random Classifier', 'Location', 'southeast');
axis square;
grid on; 
hold off; 

fprintf('AUC-ROC      : %.3f\n', AUC);
