% @=============================================================================
% Copyright (c)2025 University of Granada - SPAIN
% This software is distributed under the terms of the GNU General Public License
% as published by the Free Software Foundation. Further details on the GPLv3
% license can be found at http://www.gnu.org/copyleft/gpl.html
% 
% FOR RESEARCH PURPOSES ONLY. THE SOFTWARE IS PROVIDED "AS IS," AND THE
% UNIVERSITY OF GRANADA DO NOT MAKE ANY WARRANTY, EXPRESS OR IMPLIED, INCLUDING 
% BUT NOT LIMITED TO WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
% PURPOSE, NOR DO THEY ASSUME ANY LIABILITY OR RESPONSIBILITY FOR THE USE 
% OF THIS SOFTWARE.
%
% For more information see /license/license.html
% =============================================================================@
%
% Author: Juan Ruiz de Miras, 2025

clear all;
close all;

imageSize = 640;
patchSize = [64, 64];
overlap = 0.65; 
holdout = 0.05; 

class = {'veronese', 'non-veronese'};
nonVeronesePaintings = {'05', '06'};

% Veronese paintings for each test iteration
veronesePaintingsData = {{'01', '02', '03', '04'}, ...
                         {'02', '03', '04'}, ...
                         {'01', '03', '04'}, ...
                         {'01', '02', '04'}, ...
                         {'01', '02', '03'}};

% test paintings for each test iteration: painting '11' is the painting under
% authentication
testPaintingsData = {{'07', '08', '09', '10', '11'}, ...
                     {'01'}, ...
                     {'02'}, ...
                     {'03'}, ...
                     {'04'}};

for t = 1:size(veronesePaintingsData,2)
    % CPU and GPU seed for random calculations and reproducibility
    seed = 123456; 
    rng(seed); 
    gpurng(seed); 
    deep.gpu.deterministicAlgorithms(true);

    veronesePaintings = veronesePaintingsData{t};
    testPaintings = testPaintingsData{t};
    
    % sliding-window processing
    [patchesColorVeronese, patchesGrayscaleVeronese, patchesEdgesVeronese] = extractPatches(patchSize, overlap, class{1}, veronesePaintings);
    [patchesColorNonVeronese, patchesGrayscaleNonVeronese, patchesEdgesNonVeronese] = extractPatches(patchSize, overlap, class{2}, nonVeronesePaintings);
    [X, Y] = prepareDataCNN(patchesColorVeronese, patchesGrayscaleVeronese, patchesEdgesVeronese, ...
                            patchesColorNonVeronese, patchesGrayscaleNonVeronese, patchesEdgesNonVeronese, ...
                            patchSize);
    
    
    % holdout partition
    cv = cvpartition(size(X,4), 'HoldOut', holdout);
    XTrain = X(:, :, :, training(cv));
    YTrain = Y(training(cv));
    XVal = X(:, :, :, test(cv));
    YVal = Y(test(cv));
    
    disp('Training CNN...');
    
    % defining CNN architecture
    layers = [
        imageInputLayer([patchSize(1) patchSize(2) 5], 'Normalization', 'none')
    
        convolution2dLayer(3, 16, 'Padding', 'same')
        batchNormalizationLayer
        reluLayer
        maxPooling2dLayer(2, 'Stride', 2)
    
        convolution2dLayer(3, 32, 'Padding', 'same')
        batchNormalizationLayer
        reluLayer
        maxPooling2dLayer(2, 'Stride', 2)
    
        dropoutLayer(0.5) % regularización para evitar sobreaprendizaje por parches solapados
    
        fullyConnectedLayer(2)
        softmaxLayer
    ];
    
    % training parameters
    miniBatchSize = 128; 
    numTrainImages = size(XTrain,4);
    iterationsPerEpoch = ceil(numTrainImages / miniBatchSize);
    
    options = trainingOptions('adam', ...
                              'MaxEpochs', 100, ... 
                              'MiniBatchSize', miniBatchSize, ...
                              'Shuffle', 'every-epoch', ... 
                              'L2Regularization', 0.0005, ... % regularization to avoid overfitting from overlapping patches
                              'ValidationData', {XVal, YVal'}, ...
                              'ValidationFrequency', iterationsPerEpoch, ...
                              'ValidationPatience', 8, ... % early stopping   
                              'Metrics','accuracy', ...
                              'OutputNetwork', 'best-validation', ... 
                              'Verbose', true, ...
                              'Plots', 'none'); 
    
    % CNN training
    net = trainnet(XTrain, YTrain', layers, 'crossentropy', options);
    
    disp('Predicting probabilities for each path in test paintings...');
    
    for i=1:length(testPaintings)
        [patchesColorTest, patchesGrayscaleTest, patchesEdgesTest] = extractPatches(patchSize, overlap, 'test', {testPaintings{i}});
        [XTest, ~] = prepareDataCNN(patchesColorTest, patchesGrayscaleTest, patchesEdgesTest, ...
                                    [], [], [], ...
                                    patchSize);
    
        % computing probabilities for each patch
        score = predict(net, XTest);
        [predictedClass, ~] = scores2label(score, {'0', '1'});
        predictedClass = str2double(string(predictedClass));
    
        % statistics
        probs_veronese = score(:, 2);
        mu = mean(probs_veronese);          
        sigma = std(probs_veronese);
        N = length(probs_veronese);
        SEM = sigma / sqrt(N); % standard error of the mean
        CI_lower = mu - (1.96 * SEM); % confidence interval at 95% (Z-score = 1.96)
        CI_upper = mu + (1.96 * SEM);
        
        % show result
        numPatchesVeronese = sum(predictedClass == 1);
        numPatchesNonVeronese = sum(predictedClass == 0);
    
        disp(['Results for painting: ' testPaintings{i}]);
        disp(['Number of patches classified as Veronese = ' num2str(numPatchesVeronese)]);
        disp(['Number of patches classified as non-Veronese = ' num2str(numPatchesNonVeronese)]);
        disp(['Average Veronese probability = ' num2str(mu * 100, '%3.1f') '%']);
        fprintf('Standard Deviation (SD): %.2f\n', sigma); 
        fprintf('Confidence Interval (95%%): [%.2f%% - %.2f%%]\n', CI_lower * 100, CI_upper * 100);
        disp(' ');

        % show Veronese probability heatmap
        heatMapProbVeronese(imageSize, patchSize(1,1), overlap, score, testPaintings{i});
    end
end

