/***************************************************************
 **
 ** Fichero: lsix.c
 ** Funcion: Libreria basica de gestion de ventanas 
 **          para implementar clientes X
 ** 
 ** 	     autores: Pedro Cano Olivares, 
 **                   Antonio Lopez Fernandez, 
 **                   Carlos Urena Almagro, 
 **                   Juan Manuel Albarracin Ramirez.
 **
 **
 ***************************************************************/


#include "lsix.h"

XWMHints xwmh = 

{
    (InputHint|StateHint),
    False,
    NormalState,
    0,
    0,
    0,0,
    0,
    0,
} ;


Cursor        c1,c2,c3,c4,c5 ;

LSIX_ventana  defw= NULL  ;        /***  ventana actual        ***/
Display       *dpy = NULL ;        /***  display de trabajo    ***/

Font          fn,fn2,fn3  ;        /***  fuentes que necesito  ***/

char          fuente[]  =  "*adobe*times*bold*r*normal*120*"               ;
char          fuente2[] =  "*new century schoolbook*medium*r*normal*140*"  ;
char          fuente3[] =  "*new century schoolbook*bold*r*normal*140*"    ;

XFontStruct   *xfs, *xfs2, *xfs3 ;

/***********************************************************************/

void chkf( FILE *f, char *nombre )
{
   if (f==NULL) {
     fprintf(stderr,"\nLSIX: no se puede abrir fichero (%s)",nombre); 
     exit(1);
   }

}
/***********************************************************************/

void chk_dpy()
{
   if (dpy==NULL) {
     fprintf(stderr,"\nLSIX: no hay establecida conexion con ningun display"); 
     exit(1);
   }

}

/***********************************************************************/

void chk2()
{   
   chk_dpy();
   
   if (defw==NULL) { 
     fprintf(stderr,"\nLSIX: no hay establecida ventana de trabajo"); 
     exit(1);
   }
}

/*********************************************************************/

void chk_visual( unsigned tipo_visual, unsigned *bits_x_pixel, unsigned *clase_visual )
{
   switch( tipo_visual )
   { 
       case VISUAL_8_PLANOS :
           *bits_x_pixel= 8 ;
           *clase_visual= PseudoColor ;
           break ;
           
       case VISUAL_15_PLANOS :
           *bits_x_pixel= 15 ;
           *clase_visual= TrueColor ;
           break ;

       case VISUAL_16_PLANOS_RGB565 :
           *bits_x_pixel= 16 ;
           *clase_visual= TrueColor ;
           break ; 
     
       case VISUAL_24_PLANOS :
           *bits_x_pixel= 24 ;
           *clase_visual= TrueColor ;
           break ;

       default :
           fprintf(stderr,"error: tipo de visual desconocido\n");
           exit(1) ;
   }
} 

/*********************************************************************/

void LSIX_VentanaDeTrabajo( LSIX_ventana w )
{
  defw= w ;  
}

/*********************************************************************/
                  
void LSIX_FijarPixel( int x, int y, LSIX_Color c  )
{
  chk2();
  
  XSetForeground(dpy,defw->gc,c);    
  XDrawPoint(dpy,defw->win,defw->gc,x,y);
    
}

/*********************************************************************/
                 
LSIX_Color LSIX_LeerPixel( int x, int y  )

{ 
  XImage * imag ;
  LSIX_Color c ;
  
  chk2();  
      
  imag= XGetImage(dpy,defw->win,x,y,1,1,0xff,ZPixmap);
  c= XGetPixel( imag, 0,0 ) ;
  XDestroyImage( imag );
  
  return c ; 
}

/********************************************************************/

void LSIX_Linea( int x0, int y0, int x1, int y1, LSIX_Color c ) 
{
  chk2();
  XSetForeground(dpy,defw->gc,c);
  XDrawLine(dpy,defw->win,defw->gc,x0,y0,x1,y1);
}

/*******************************************************************/

int LSIX_Ancho()
{
  chk2();
  return defw->resx ;
}

/******************************************************************/

int LSIX_Alto()
{
  chk2();
  return defw->resy ;
}  

/*******************************************************************/

void LSIX_Sincroniza()
{
  chk2();

  XSync(dpy,True) ;   
}

/**********************************************************************/

void LSIX_AbrirDisplay( char * nombre_disp, int modo_sinc )

{

    int tipo_visual ;

     /**** falta: cerrar display y ventanas si dpy != NULL ***/

    if ( (dpy=XOpenDisplay(nombre_disp)) == NULL ) {
      fprintf (stderr,"\nLSIX: no se puede conectar con el display %s.",XDisplayName(nombre_disp));
      exit(1);
    }
    XSync(dpy,False);
         
    if (modo_sinc)
       XSynchronize(dpy,True);
}


/****************************************************************************/

void LSIX_CargarPaleta( int entrada, int *rojo, int *verde, int *azul )
{         
   XColor xcolor ;

   chk2() ;

   xcolor.pixel= entrada;

   XQueryColor(dpy,defw->map,&xcolor) ;
  
   *rojo = xcolor.red / 255 ;      
   *verde= xcolor.green / 255 ;
   *azul= xcolor.blue / 255 ;                                 
}

/****************************************************************************/

void LSIX_CambiarPaleta( byte entrada, byte r, byte g, byte b )
{         
   XColor xcolor ;

   chk2() ;

   xcolor.flags= DoRed | DoGreen | DoBlue ;

   xcolor.pixel= entrada ;
   xcolor.red=   r*255 ;      
   xcolor.green= g*255 ;
   xcolor.blue=  b*255 ;
        
   XStoreColor(dpy,defw->map,&xcolor) ;
    
   defw->xswa.colormap=  defw->map;     
   XChangeWindowAttributes(dpy,defw->win,CWColormap,&(defw->xswa)) ;                         
}

/****************************************************************************/

LSIX_Color LSIX_CreaColorRGB( unsigned char r, unsigned char g, unsigned char b )
{
  
  chk2() ;
  
  switch ( defw->tipo_visual )
  {
     case VISUAL_8_PLANOS :
     
         return 128 + (LSIX_Color)((1.0/6.0)*(r+g+b)) ;
     
     case VISUAL_15_PLANOS :
     
          return   ( ((LSIX_Color)r>>3) <<  0L )  
                 + ( ((LSIX_Color)g>>3) <<  5L ) 
                 + ( ((LSIX_Color)b>>3) << 10L ) ;

     case VISUAL_16_PLANOS_RGB565 :

          return   ( ((LSIX_Color)r>>3) << 11L )  
                 + ( ((LSIX_Color)g>>2) <<  5L ) 
                 + ( ((LSIX_Color)b>>3) <<  0L ) ;
                    
     case VISUAL_24_PLANOS : 
         
          return   ( ((LSIX_Color)r) <<  0L )  
                 + ( ((LSIX_Color)g) <<  8L ) 
                 + ( ((LSIX_Color)b) << 16L ) ;

     default :
           fprintf(stderr,"error: tipo de visual desconocido\n");
           exit(1) ;
  }

  return (LSIX_Color)0 ; 

}

void LSIX_LeeComponentesRGB( LSIX_Color c, byte *r, byte *g, byte *b )
{
  chk2() ;
  
  switch ( defw->tipo_visual )
  {
     case VISUAL_8_PLANOS :
     
         *r= *g= *b= (byte) c ;
         break ;

     case VISUAL_16_PLANOS_RGB565 :

         *r= (byte)(((c>>11L) & 0x1F ) <<3 ) ;
         *g= (byte)(((c>> 5L) & 0x3F ) <<2 ) ;
         *b= (byte)(((c>> 0L) & 0x1F ) <<3 ) ;
         break ;


     case VISUAL_15_PLANOS :
     
         *r= (byte)(((c>> 0L) & 0x1F ) <<3 ) ;
         *g= (byte)(((c>> 5L) & 0x1F ) <<3 ) ;
         *b= (byte)(((c>>10L) & 0x1F ) <<3 ) ;
         break ;

     case VISUAL_24_PLANOS :
     
         *r= (byte)((c>> 0L) & 0xFF ) ;
         *g= (byte)((c>> 8L) & 0xFF ) ;
         *b= (byte)((c>>16L) & 0xFF ) ;
         break ;
                    
  }

} 

/****************************************************************************/

LSIX_ventana LSIX_AbrirVentana ( int iresx, int iresy, const char *titulo, int visible, int tipo_visual )


{   int           i             ;
    LSIX_ventana  w             ;
    XEvent        event, 
                  report        ;
    Colormap      mapa_anterior ;    /** mapa que habia antes de abrir la ventana **/
   
    int           cuenta= 0     ;
   
    unsigned int  bits_x_pixel  ;    /** 8/24                    **/
    unsigned int  clase_visual  ;    /** PseudoColor/TrueColor   **/
    unsigned int  bit_colormap  ;    /** 0 si no cambia colormap **/
    

    XVisualInfo  vinfo ;
    XRectangle   rect   ;

    XGCValues    gc_values ;
    Pixmap       gc_pixmap ;
    
    chk_dpy() ;
    
    if ((w=(LSIX_ventana)malloc(sizeof(struct xventana)))==NULL) 
    {  fprintf(stderr,"\nLSIX: imposible reservar memoria (en abrir_ventana)");
       exit(1);
    }
         
    w->resx= iresx ;
    w->resy= iresy ;

   /****************** crear una WINDOW ***********/


   /*** Comprobacion de que existe un visual del tipo requerido
   ***/

   chk_visual( tipo_visual, &bits_x_pixel,&clase_visual ) ;

   #ifdef DEBUG
   printf("\nVisual ID (por defecto) == %x hex.\n\n",
             XVisualIDFromVisual(DefaultVisual(dpy,DefaultScreen(dpy))) ) ;
   #endif

   if ( ! XMatchVisualInfo(dpy,
             DefaultScreen(dpy),bits_x_pixel,clase_visual,&vinfo) )
   {
       printf("\nError: este display no soporta ningun"
              "\nvisual del tipo indicado en 'LSIX_AbrirVentana'\n") ;
       exit(1) ;
   }

   #ifdef DEBUG
   else
       printf("\n Encontrado el visual del tipo requerido, ID == %x hex.\n\n",vinfo.visualid);
       printf("\n vinfo.depth = %d\n",vinfo.depth);
   #endif

   printf("LSIX_AbrirVentana: visual de %d bits/pixel\n",bits_x_pixel) ;
     
    /************************ crear un PIXMAP *****/
   
    if (!visible) 
    {    
    
      Pixmap pm ;
      
      w->visible= FALSE ;
      pm= XCreatePixmap(dpy,XDefaultRootWindow(dpy),
                        w->resx, w->resy, 
                        bits_x_pixel
                       ) ;
      w->win= pm ;
      w->gc= XCreateGC( dpy, w->win, 0, NULL ) ; 
      w->tipo_visual= tipo_visual ;

      return w ;
    }
   
   
   /** Crear un Colormap para ventanas de 256 colores (8 planos)
   *** (usando el visual encontrado antes)
   **/

   if ( tipo_visual == VISUAL_8_PLANOS )
   {
      w->map = XCreateColormap (dpy,DefaultRootWindow(dpy),
                                vinfo.visual,
                                AllocAll) ; 
  

      /** Fijar la paleta del nuevo colormap
      ***  (copiandola en parte del ya existente)
      **/

      mapa_anterior= XDefaultColormap(dpy,DefaultScreen(dpy)) ;
     
      for ( i=0; i<256 ; i++ )
      {      
         w->xcolor[i].pixel=i;
         w->xcolor[i].flags= DoRed | DoGreen | DoBlue;
      
         if (i<128) 
         {  XQueryColor(dpy,mapa_anterior,(w->xcolor+i)) ;           
         }
         else 
         {  unsigned int j= 2*(i-128) ; 
            w->xcolor[i].red=   j*255 ;      
            w->xcolor[i].green= j*255 ;
            w->xcolor[i].blue=  j*255 ;
         }
        
         XStoreColor(dpy,w->map,(w->xcolor)+i) ;
      }
   
      XStoreColors(dpy,w->map,w->xcolor,256) ;  /** redundante ?? **/   
      XSync(dpy,False);      

   }
   else  /** no alojar colores para 24 bits **/
   {
     w->map = XCreateColormap (dpy,DefaultRootWindow(dpy),vinfo.visual,AllocNone) ;
   }

   

   /*** Definir los atributos de la nueva ventana y crearla
   **/

   w->visible= TRUE ;

   w->bw  = 1 ;
   w->pad = 1 ;

   w->xsh.flags  = (PPosition|PSize) ;
   w->xsh.height = w->resy           ;
   w->xsh.width  = w->resx           ;

   w->xsh.x = (DisplayWidth(dpy,DefaultScreen(dpy))  - w->xsh.width)  / 2   ;
   w->xsh.y = (DisplayHeight(dpy,DefaultScreen(dpy)) - w->xsh.height) / 2   ;

   w->xswa.background_pixel  =  0L            ; 
   w->xswa.bit_gravity       =  StaticGravity ;
   w->xswa.win_gravity       =  NorthWestGravity ;
   w->xswa.backing_store     =  Always        ;
   w->xswa.backing_planes    =  AllPlanes     ;
   w->xswa.save_under        =  True          ;
   w->xswa.override_redirect =  False         ;
   w->xswa.colormap          =  w->map        ; 

   w->tipo_visual= tipo_visual ;   
    
   w->win= XCreateWindow
           (    dpy, 
                DefaultRootWindow(dpy),
                w->xsh.x,
                w->xsh.y,
                w->xsh.width,
                w->xsh.height,
                0, 
                vinfo.depth, 
                InputOutput,
                vinfo.visual,
                CWBackPixel|
                CWBorderPixel|
                CWColormap|
                CWBitGravity|
                CWWinGravity|
                CWBackingStore|
                CWBackingPlanes|
                CWSaveUnder|
                CWOverrideRedirect,             
                &(w->xswa)
           ) ;
                   
   if (!(w->win)) 
   { fprintf(stderr,"LSIX: Imposible crear ventana") ;
     exit(1) ;
   }

   XSync(dpy,False);
                      
   /**  Crear el "Graphics Context" a usar con esta ventana.
   ***  (todos los atributos toman valores por defecto, excepto
   ***  los que se cambian explicitamente despues)
   **/

   w->gc= XCreateGC( dpy, w->win, 0, NULL ) ;

   gc_values.graphics_exposures = False ;
   XChangeGC( dpy, w->gc, 
              GCGraphicsExposures, 
              &gc_values ) ;

   /**  Poner el titulo de la ventana
   **/

   XSetStandardProperties(dpy,w->win,titulo,titulo,None,NULL,0,&(w->xsh));
   XSetWMHints(dpy,w->win,&xwmh);
   XSync(dpy,False);

   /** indicar cuales son los eventos que se van a aceptar
   *** en esta ventana
   **/
   
   XSelectInput(dpy,w->win,ExposureMask|StructureNotifyMask|
                           PointerMotionMask|ButtonPressMask|
                           ButtonReleaseMask|KeyPressMask);       
   XSync(dpy,False);
   
   /*** dibujar la ventana en el display
   ***/

   XMapRaised(dpy,w->win);

    /** Esperar hasta que se reciba un evento de notificacion 
    *** de que la ventana se ha dibujado (MapNotify)
    **/

    do
    {  XNextEvent(dpy, &report);
    } 
    while ( report.type != MapNotify ) ;   
     
    XSync(dpy,False);

    /** borrar el fondo 
    **/
     
    XClearWindow (dpy,w->win);
    XSetForeground(dpy,w->gc,255);

    /** Cargar las fuentes que voy a utilizar 
    **/
    
    fn  = XLoadFont (dpy,fuente)  ;  xfs  = XQueryFont (dpy,fn)  ;
    fn2 = XLoadFont (dpy,fuente2) ;  xfs2 = XQueryFont (dpy,fn2) ;
    fn3 = XLoadFont (dpy,fuente3) ;  xfs3 = XQueryFont (dpy,fn3) ;

    XSetFont(dpy,w->gc,fn3); /* fuente por defecto */
    
    /** Definicion de Cursor 
    **/

    c1= XCreateFontCursor (dpy,XC_top_left_arrow);
    c2= XCreateFontCursor (dpy,XC_tcross);
    c3= XCreateFontCursor (dpy,XC_ul_angle);
    c4= XCreateFontCursor (dpy,XC_lr_angle);
    c5= XCreateFontCursor (dpy,XC_watch);

    XDefineCursor (dpy,w->win,c1); /* cursor por defecto */

    return w ;
 }

/*****************************************************************/

void LSIX_CambiaPunteroRaton(int tipo)
{ 
  chk2(); 
 
  switch (tipo)
  { case FLECHA: 
       XDefineCursor (dpy,defw->win,c1); 
       break;  
    case ESI: 
       XDefineCursor (dpy,defw->win,c3); 
       break;
    case EID: 
       XDefineCursor (dpy,defw->win,c4); 
       break;
    case CRUZ: 
       XDefineCursor (dpy,defw->win,c2); 
       break;
    case RELOJ: 
       XDefineCursor (dpy,defw->win,c5); 
       break;
    default: 
       XUndefineCursor (dpy,defw->win);
  }
}

/*************************************************************************/

void LSIX_FijarFont (int tipo)
{  
  chk2() ;

  switch (tipo) 
  {  case 1 : 
         XSetFont (dpy,defw->gc,fn); 
         break ; 
     case 2 : 
         XSetFont (dpy,defw->gc,fn2); 
         break ;
     case 3 : 
         XSetFont (dpy,defw->gc,fn3); 
         break ;
     default:
         fprintf(stderr,"Tipo de font desconocido en LSIX_FijarFont\n");
	 exit(1);
  }
 } 

/*************************************************************************/

void LSIX_PintaCadena (char *cadena, int posx, int posy)
{  
  chk2();
  XDrawString (dpy,defw->win,defw->gc,posx,posy,cadena,strlen(cadena));
}  

/*************************************************************************/

int LSIX_StringSizeFont (char *cadena, int tipo)
{  
  int l;  
  
  chk2();
  switch (tipo) 
  {  case 1:  
          l= XTextWidth (xfs,cadena,strlen(cadena)); 
          break ;
     case 2:  
          l= XTextWidth (xfs2,cadena,strlen(cadena)); 
          break ;
     case 3:  
          l= XTextWidth (xfs3,cadena,strlen(cadena)); 
          break;
  }

  return l ;
}
 
/*************************************************************************/

void LSIX_EsperaAccionRaton( unsigned *accion, unsigned *x, unsigned  *y )
{
  XEvent evento ;
  
  chk2();

  if (!defw->visible) {
    fprintf(stderr,"\nLSIX: no se pueden pedir posicion del raton en ventanas no visibles");
    exit(1);
  }
  
  XWindowEvent(dpy,defw->win,PointerMotionMask|ButtonPressMask|ButtonReleaseMask,&evento);
  
  switch (evento.type) {
    
    case MotionNotify :
      *accion= MOVER ;
      *x= evento.xmotion.x ;
      *y= evento.xmotion.y ;
      return ;
      
    case ButtonPress :
      *accion= PULSAR ;
      *x= evento.xbutton.x ;
      *y= evento.xbutton.y ; 
      return ;  
      
    case ButtonRelease :
      *accion= LEVANTAR ;
      *x= evento.xbutton.x ;
      *y= evento.xbutton.y ;
      return ;  
  }

}

/*************************************************************************/

boolean  LSIX_EsperaEventoExposicion()
{
   XEvent evento ;
  
  chk2();

  if (!defw->visible) {
    fprintf(stderr,"\nLSIX: no se pueden esperar eventos en ventanas no visibles");
    exit(1);
  }
  
  XWindowEvent(dpy,defw->win,ExposureMask|ButtonPressMask,&evento);
  
  return evento.type != ButtonPress ;

}

/*************************************************************************/

void LSIX_InformaRaton( int *accion, int *x, int *y )
{
  int rx,ry;
  unsigned int kb;
  Window root,child;

  chk2();

  if (!defw->visible) {
    fprintf(stderr,"\nLSIX: no se pueden pedir posicion del raton en ventanas no visibles");
    exit(1);
  }

  *accion=LEVANTAR;

  if (!XQueryPointer(dpy,defw->win,&root,&child,&rx,&ry,x,y,&kb))
  { *accion= -1 ; 
    *x= *y= 0 ;  
  }
  else  
    if (kb & Button1Mask)
       *accion=PULSAR; 

}

/*************************************************************************/

void LSIX_EsperaTeclaPulsada ( int *codigo, unsigned int *tecla )
{
  XEvent         evento ;
  KeySym         key ;
  XComposeStatus cs  ;
  char           cadena[20] ;  
  int            tc=20,cont;
  
  chk2();

  if (!defw->visible)
  {  fprintf(stderr,"\nLSIX: no se pueden hacer esperas de teclas en ventanas no visibles");
    exit(1);
  }

   *codigo=ERRT;
   XWindowEvent(dpy,defw->win,KeyPressMask,&evento) ;

   switch (evento.type) 
   {
     case KeyPress :
        
        cont= XLookupString (&evento.xkey,cadena,tc,&key,&cs);
        cadena[cont]= '\0';
        *tecla= cadena[0];
	  
        switch (key)
        {  
	   case XK_BackSpace:
           case XK_Delete: 
	   
              *codigo=BORRAR ; 
              break ;
	      
           case XK_Return:
           case XK_KP_Enter:
	    
	      *codigo=ENTER ; 
	      break ;
	      
           case XK_Shift_L: 
           case XK_Shift_R: 
	   
	      *codigo=SHIFT ; 
	      break ;
			   
	   case XK_Escape: 
	   
	      *codigo=ESCAPE ; 
	      break ;
                           
	   case XK_Tab: 
	    
	      *codigo=TAB ; 
	      break ;
	      
           case XK_Up: 
	   
	      *codigo=UP ;
	      break ;
	      
	   case XK_Down : 
	   
	      *codigo=DOWN ; 
	      break ;
                          
           case XK_Right:
	   
	      *codigo=RIGHT ; 
	      break ;
	      
	   case XK_Left: 
	   
	      *codigo=LEFT ; 
	      break ;
        }                    
        break;
   }
 }

/*****************************************************************/

long LSIX_ImageSize (int x0,int y0,int x1,int y1)
{ 
  XImage *x;
  long t;

  x = XGetImage(dpy,defw->win,x0,y0,x1-x0+1,y1-y0+1,0xFF,ZPixmap); 
  t = (x->bytes_per_line*(y1-y0+1)*sizeof(char)) + sizeof(XImage);  
  XDestroyImage (x);
  return (t);
}

/******************************************************************/

void LSIX_GrabarVentana(int xi,int yi,int xf,int yf,void *buffer)
{ 
  XImage *xima;
  long ti,tx;
  int ph,pv;

  chk2(); 
  ph= xf-xi+1;
  pv= yf-yi+1;
  xima= XGetImage(dpy,defw->win,xi,yi,ph,pv,0xFF,ZPixmap);
  tx=sizeof(XImage); 
  ti=xima->bytes_per_line*pv*sizeof(char);
  memcpy (buffer,xima,tx);
  memcpy ((void *)(((byte *)buffer)+tx),xima->data,ti);
  XDestroyImage (xima);
}

/*****************************************************************/

void LSIX_RestauraVentana(int xi,int yi,void *buffer,int t)
{ 
  XImage xima;   
  long tx,ti;

  chk2();
  tx=sizeof(XImage); 
  memcpy (&xima,buffer,tx);
  ti=sizeof(char)*xima.bytes_per_line*xima.height;
  xima.data=(char *)malloc (ti);
  memcpy (xima.data,(void *)(((byte *)buffer)+tx),ti) ;   
  XPutImage(dpy,defw->win,defw->gc,&xima,0,0,xi,yi,xima.width,xima.height);
  free (xima.data);
}

/**********************************************************/

void LSIX_CargaImagen( boolean visible, unsigned char *img[256], int xini, int yini )

{  XImage *ima ;
   unsigned char *buffer ;
   int  nx=256,ny=256,x,y,yy;
   long int tbuffer;
      
   tbuffer= (long)nx*(long)ny ;  

   if (NULL==(buffer=(unsigned char *)malloc(tbuffer*sizeof(unsigned char)))) 
   {
     fprintf(stderr,"LSIX: imposible reservar memoria (en LSIX_CargaImagen)");
     exit(1);
   }
   
   /***   Transformo imagen original a buffer antes de pintarlo 
   ***/

   for (y=0;y<ny;y++)
   {
     yy=256*y;
     for (x=0;x<ny;x++)
         buffer[yy+x]=128+(img[y][x]>>1);
   }

   ima= XCreateImage (dpy,DefaultVisual(dpy,DefaultScreen(dpy)),8,
                      ZPixmap,0,
                      (char *)buffer,
                      nx,ny,8,nx);
   
   XPutImage(dpy,defw->win,defw->gc,ima,0,0,xini,yini,nx,ny);
   
   XDestroyImage(ima) ; 
                                          
} 

/************************************************************************/

void LSIX_RellenaRectangulo( int x0, int y0, int x1, int y1, LSIX_Color c )
{       
    chk2() ;
    XSetForeground(dpy,defw->gc,c) ;
    XFillRectangle (dpy,defw->win,defw->gc,x0,y0,x1-x0+1,y1-y0+1) ;
}  

/***************************************************************************/

void LSIX_RellenaPoligono( int n, int x0, int y0, int *puntos, LSIX_Color c ) 
{
    XPoint *tpuntos ;
    int     i, *p ;

    chk2() ;

    RESERVA( tpuntos, n, XPoint, "RellenaPoligono" ) ;
     
    p= puntos ;
  
    for( i= 0 ; i<n ; i++ ) {
       tpuntos[i].x= x0 + *(p++) ;
       tpuntos[i].y= y0 + *(p++) ;
    }
 
    XSetForeground(dpy,defw->gc,c) ;
    XFillPolygon(dpy,defw->win,defw->gc,tpuntos,n,Complex,CoordModeOrigin) ;
 
    free(tpuntos) ;
    
}

/************************************************************************/

void LSIX_DibujaRectangulo( int x0, int y0, int x1, int y1, LSIX_Color c )
{
    
    XRectangle rect ;

    chk2() ;

    rect.x=       x0 ;
    rect.y=       y0 ;
    rect.width=   x1 - x0 + 1 ;
    rect.height=  y1 - y0 + 1 ;
    
    XSetForeground(dpy,defw->gc,c) ;
    XDrawRectangles(dpy,defw->win,defw->gc,&rect,1) ;
}  

/************************************************************************/

void LSIX_FijarRectanguloRecorte( int x0, int y0, int x1, int y1 )

{
    XRectangle rect ;

    chk2() ;

    rect.x=       x0 ;
    rect.y=       y0 ;
    rect.width=   x1 - x0 + 1 ;
    rect.height=  y1 - y0 + 1 ;
    
    XSetClipRectangles(dpy,defw->gc,0,0,&rect,1,Unsorted) ;
}

/************************************************************************/

void LSIX_CerrarVentana()
{
  chk2();
  /** printf("Destruyendo ventana .....\n"); **/

  if (defw->visible) 
    XDestroyWindow(dpy,defw->win);
  else
    XFreePixmap(dpy,defw->win);
   
  /** Descargar las fuentes creadas anteriormente 
  **/
  XUnloadFont (dpy,fn);
  XUnloadFont (dpy,fn2);
  XUnloadFont (dpy,fn3);

  /** descargar los cursores 
  **/
  XFreeCursor (dpy,c1);
  XFreeCursor (dpy,c2);
  XFreeCursor (dpy,c3);
  XFreeCursor (dpy,c4);
  XFreeCursor (dpy,c5);

  free(defw) ;
   
  /** printf("Ya esta.\n") ; **/ 
  defw= NULL ;
}

/************************************************************************/

void LSIX_CopiarVentana( LSIX_ventana org, int desx, int desy )
{
  chk2() ; 

  if (org==NULL) 
  { printf("\nLSIX error: ventana de origen incorrecta (CopiarArea)");
    exit(1);
  }
  
  XCopyArea( dpy,org->win,defw->win,defw->gc,0,0,org->resx,org->resy,desx,desy ) ;
}

/************************************************************************/

void LSIX_FijarColorActual( LSIX_Color c )
{
    XSetForeground(dpy,defw->gc,c) ;
}  

/***************************************************************************/

void LSIX_Bell ( char valor )
{
    XBell (dpy,(int)valor);  /* valor debe de estar entre -100 y 100 */
}

/***************************************************************************/


void LSIX_ClearArea (int xi,int yi,int xf,int yf)
{
    XClearArea (dpy,defw->win,xi,yi,xf-xi+1,yf-yi+1,False);    
}


/***************************************************************************/

void LSIX_ClearWindow (void)
{
    XClearWindow (dpy,defw->win);    
}

/***************************************************************************/

boolean LSIX_SoportaVisual( unsigned tipo_visual )
{
  unsigned int bits_x_pixel, clase_visual ;
  XVisualInfo vinfo ;
   
  chk_dpy() ;
  chk_visual( tipo_visual, &bits_x_pixel, &clase_visual ) ;

  if ( XMatchVisualInfo(dpy,
             DefaultScreen(dpy),bits_x_pixel,clase_visual,&vinfo) )  
      return TRUE ;

  return FALSE ;
 
}

/***************************************************************************/

unsigned LSIX_LeeMejorVisual( )
{
    chk_dpy() ;

    if ( LSIX_SoportaVisual( VISUAL_24_PLANOS ) )
        return VISUAL_24_PLANOS ;

    if ( LSIX_SoportaVisual( VISUAL_16_PLANOS_RGB565 ) )
        return VISUAL_16_PLANOS_RGB565 ;
        
    if ( LSIX_SoportaVisual( VISUAL_15_PLANOS ) )
        return VISUAL_15_PLANOS ;

    if ( LSIX_SoportaVisual( VISUAL_8_PLANOS ) )
        return VISUAL_8_PLANOS ;

    fprintf(stderr,"error en 'LSIX_LeeMejorVisual' :  no encuentro un visual soportado\n");
    exit(1);

    return VISUAL_8_PLANOS ;
}
