/*****************************************************************************
** 
**   MODULE NAME  SphTri.h++   (header file for Spherical Triangles)
**
**   DESCRIPTION
**
**      This file defines the SphericalTriangle class definition, which
**      defines member functions for Monte Carlo sampling, point containment,
**      and other basic operations on spherical triangles.
**
**
**   HISTORY
**      Name    Date        Description
**
**      Carlos  18/11/96    Adapted to use the vector library in RT
**      arvo    04/08/95    Further optimized sampling algorithm.
**      arvo    10/11/94    Added analytic sampling algorithm.
**      arvo    06/14/94    Initial implementation.
**
**
**   (c) Copyright 1995
**       James Arvo and California Institute of Technology, Pasadena, CA
**   (c) Copyright 1994, 1995
**       Program of Computer Graphics, Cornell University, Ithaca, NY
**       ALL RIGHTS RESERVED
**
*****************************************************************************/

#ifndef SPHTRI_INCLUDED
#define SPHTRI_INCLUDED

#include <stdlib.h>
#include <math.h>
#include <Vector.H>

/*
 *  The Spherical Triangle ABC.  Edge lengths (segments of great circles)
 *  are a, b, and c.  The (dihedral) angles are Alpha, Beta, and Gamma.
 *
 *         B
 *           o
 *           | \
 *           |   \
 *           |Beta \
 *           |       \
 *           |         \ a
 *         c |           \ 
 *           |             \
 *           |               \
 *           |                 \
 *           |                   \
 *           |Alpha         Gamma  \
 *           o----------------------o
 *         A            b              C
 *
 */

#define ArcCos acos
#define ArcSin asin
#define Sqrt   sqrt
#define Pi     M_PI

#define Unit(V)            ((V).Normalized())
#define Clamp(min,x,max)   MIN(MAX(x,min),max)
#define Sign(x)            (((x)<0.0)?-1.0:+1.0)

#define MIN(a,b)        ((a)<(b)?(a):(b))
#define MAX(a,b)        ((a)>(b)?(a):(b))

inline Vector operator/(const Vector &v1, const Vector &v2 )
{
    return v1 - (v1|v2)*v2 ;
}


class SphericalTriangle 
{
    public:
        SphericalTriangle() { Init(); }
        SphericalTriangle( const SphericalTriangle &T ) { *this = T; }
        SphericalTriangle( const Vector &, const Vector &, const Vector & );
        SphericalTriangle & operator()( const Vector &, const Vector &, const Vector & );
       ~SphericalTriangle( ) {}
        void   operator=( const SphericalTriangle &T ) { *this = T; }
        Vector   Sample   ( Scalar x, Scalar y ) const;  // Fast Analytic inversion
        void   Coord    ( const Vector &P , Scalar & u, Scalar & v    ) const;  // Get 2D coords of a point.
        int    Orient( ) const { return orient; }
        int    Inside( const Vector & ) const;
        Scalar  SolidAngle() const { return area; }
        Scalar  SignedSolidAngle() const { return -orient * area; } // CC is pos.
        const  Vector &A()  const { return A_       ; }
        const  Vector &B()  const { return B_       ; }
        const  Vector &C()  const { return C_       ; }
        Scalar  a()        const { return a_       ; }
        Scalar  b()        const { return b_       ; }
        Scalar  c()        const { return c_       ; }
        Scalar  Cos_a()    const { return cos_a    ; }
        Scalar  Cos_b()    const { return cos_b    ; }
        Scalar  Cos_c()    const { return cos_c    ; }
        Scalar  Alpha()    const { return alpha    ; }
        Scalar  Beta ()    const { return beta     ; }
        Scalar  Gamma()    const { return gamma    ; }
        Scalar  CosAlpha() const { return cos_alpha; }
        Scalar  CosBeta () const { return cos_beta ; }
        Scalar  CosGamma() const { return cos_gamma; }

    private: // functions

        void Init( );
        void Init( const Vector &A, const Vector &B, const Vector &C );

    private: // data

        Vector  A_, B_, C_, U;
        Scalar a_, b_, c_;
        Scalar alpha, beta, gamma;
        Scalar cos_a, cos_b, cos_c;
        Scalar cos_alpha, cos_beta, cos_gamma;
        Scalar area;
        Scalar sin_alpha, product;  // Used in sampling algorithm.
        int   orient;
};

extern SphericalTriangle Dual( const SphericalTriangle &T );

inline double DihedralAngle( const Vector &A, const Vector &B, const Vector &C )
{
    return ArcCos( Unit( A * B ) | Unit( C * B ) );
}

inline double CosDihedralAngle( const Vector &A, const Vector &B, const Vector &C )
{
    Scalar x = Unit( A * B ) | Unit( C * B );
    if( x < -1.0 ) x = -1.0;
    if( x >  1.0 ) x =  1.0;
    return x;
}

#endif




