/*****************************************************************************
** 
**   MODULE NAME  SphTri.C   (Spherical Triangle)
**
**   DESCRIPTION
**
**      This module implements the class SphericalTriangle.
**
**
**   HISTORY
**      Name    Date        Description
**
**      Carlos  18/11/96    Adapted to use the Vector library in RT
**      arvo    04/08/95    Further optimized sampling algorithm.
**      arvo    10/11/94    Added analytic sampling algorithm.
**      arvo    06/14/94    Initial implementation.
**
**
**   (c) Copyright 1995
**       James Arvo and California Institute of Technology, Pasadena, CA
**   (c) Copyright 1994, 1995
**       Program of Computer Graphics, Cornell University, Ithaca, NY
**       ALL RIGHTS RESERVED
**
*****************************************************************************/

#include <SphTri.H>
#include <stdio.h>
#include <iostream.h>
#include <math.h>
#include <Vector.H>


SphericalTriangle::SphericalTriangle
( 
   const Vector &A0, 
   const Vector &B0, 
   const Vector &C0 
)

{
    Init( A0, B0, C0 );
}

/*-------------------------------------------------------------------------*
 *                                                                         *
 * Initialize all fields.  Create a null spherical triangle.               *
 *                                                                         *
 *-------------------------------------------------------------------------*/

void 
SphericalTriangle::Init
(
)

{
    a_ = 0;   cos_alpha = 0;  cos_a = 0;  alpha = 0;  
    b_ = 0;   cos_beta  = 0;  cos_b = 0;  beta  = 0;  
    c_ = 0;   cos_gamma = 0;  cos_c = 0;  gamma = 0; 

    A_ = Vector(0,0,0) ;
    B_ = Vector(0,0,0) ;
    C_ = Vector(0,0,0) ;
 
    area      = 0;
    orient    = 0;
    sin_alpha = 0;
    product   = 0;
    U         = Vector(0,0,0) ;
}


/*-------------------------------------------------------------------------*
 *                                                                         *
 *  Construct the spherical triange from three vertices.  Assume that the  *
 *  sphere is centered at the origin.  The vectors A, B, and C need not    *
 *  be normalized.                                                         *
 *                                                                         *
 *-------------------------------------------------------------------------*/


void 
SphericalTriangle::Init
( 
   const Vector &A0, 
   const Vector &B0, 
   const Vector &C0 
)

{
    // Normalize the three vectors -- these are the vertices.

    A_ = Unit( A0 ) ;
    B_ = Unit( B0 ) ;
    C_ = Unit( C0 ) ;

    // Compute and save the cosines of the edge lengths.

    cos_a = B_ | C_;
    cos_b = A_ | C_;
    cos_c = A_ | B_;

    // Compute and save the edge lengths.

    a_ = ArcCos( cos_a );
    b_ = ArcCos( cos_b );
    c_ = ArcCos( cos_c );

    // Compute the cosines of the internal (i.e. dihedral) angles.

    cos_alpha = CosDihedralAngle( C_, A_, B_ );
    cos_beta  = CosDihedralAngle( A_, B_, C_ );
    cos_gamma = CosDihedralAngle( A_, C_, B_ );

    // Compute the (dihedral) angles.

    alpha = ArcCos( cos_alpha );
    beta  = ArcCos( cos_beta  );
    gamma = ArcCos( cos_gamma );

    // Compute the solid angle of the spherical triangle.

    area = alpha + beta + gamma - M_PI ;

    // Compute the orientation of the triangle.

    orient = (int) Sign( A_ | ( B_ * C_ ) );

    // Initialize three variables that are used for sampling the triangle.

    U         = Unit( C_ / A_ );  // In plane of AC orthogonal to A.
    sin_alpha = sin( alpha );
    product   = sin_alpha * cos_c;

}


/*-------------------------------------------------------------------------*
 *                                                                         *
 *  Construct the spherical triange from three vertices.  Assume that the  *
 *  sphere is centered at the origin.  The vectors A, B, and C need not    *
 *  be normalized.                                                         *
 *                                                                         *
 *-------------------------------------------------------------------------*/

SphericalTriangle & 
SphericalTriangle::operator()
(  
   const Vector &A0, 
   const Vector &B0, 
   const Vector &C0 
)

{
    Init( A0, B0, C0 );
    return *this;
}

/*-------------------------------------------------------------------------*
 * I N S I D E                                                             *
 *                                                                         *
 * Determine if the vector W is inside the triangle.  W need not be a      *
 * unit vector                                                             *
 *                                                                         *
 *-------------------------------------------------------------------------*/

int SphericalTriangle::Inside
(  
   const Vector &W 
)  
   const

{
    Vector V = (Scalar) Orient() * W  ; 

    if( ( V | ( A() * B() )) < 0.0 ) return 0;
    if( ( V | ( B() * C() )) < 0.0 ) return 0;
    if( ( V | ( C() * A() )) < 0.0 ) return 0;

    return 1;
}

/*-------------------------------------------------------------------------*
 * S A M P L E                                                             *
 *                                                                         *
 * Generate samples from the current spherical triangle.  If x1 and x2 are *
 * random variables uniformly distributed over [0,1], then the returned    *
 * points are uniformly distributed over the solid angle.                  *
 *                                                                         *
 *-------------------------------------------------------------------------*/

Vector 
SphericalTriangle::Sample
( 
  Scalar x1 , 
  Scalar x2 
) 
  const
    
{
    // Use one random variable to select the area of the sub-triangle.
    // Save the sine and cosine of the angle phi.

    register Scalar phi = x1 * area - Alpha();
    register Scalar s   = sin( phi );
    register Scalar t   = cos( phi );

    // Compute the pair (u,v) that determines the new angle beta.

    register Scalar u = t - cos_alpha;
    register Scalar v = s + product  ;  // sin_alpha * cos_c

    // Compute the cosine of the new edge b.

    Scalar q = ( cos_alpha * ( v * t - u * s ) - v ) / 
              ( sin_alpha * ( u * t + v * s )     );

    // Compute the third vertex of the sub-triangle.

    Vector C_new = q * A() + Sqrt( 1.0 - q * q ) * U;

    // Use the other random variable to select the height z.

    Scalar z = 1.0 - x2 * ( 1.0 - (C_new | B()) );

    // Construct the corresponding point on the sphere.

    Vector D = C_new / B();  // Remove B component of C_new.

    return z * B() + Sqrt( ( 1.0 - z * z ) / ( D | D ) ) * D;
    
}


/*-------------------------------------------------------------------------*
 * C O O R D                                                               *
 *                                                                         *
 * Compute the two coordinates (x1,x2) corresponding to a point in the     *
 * spherical triangle.  This is the inverse of "Sample".                   *
 *                                                                         *
 *-------------------------------------------------------------------------*/

void
SphericalTriangle::Coord
( 
   const Vector &P1 ,
   Scalar & u ,
   Scalar & v
) 
   const

{
    Vector P = Unit( P1 );

    // Compute the new C vertex, which lies on the arc defined by B-P
    // and the arc defined by A-C.

    Vector C_new = Unit( ( B() * P ) * ( C() * A() ) );

    // Adjust the sign of C_new.  Make sure it's on the arc between A and C.

    if( (C_new | ( A() + C() )) < 0.0 ) 
        C_new *= -1.0 ;

    // Compute x1, the area of the sub-triangle over the original area.

    Scalar cos_beta  = CosDihedralAngle( A(), B(), C_new  );
    Scalar cos_gamma = CosDihedralAngle( A(), C_new , B() );
    Scalar sub_area  = Alpha() + acos( cos_beta ) + acos( cos_gamma ) - Pi;
    Scalar x1        = sub_area / SolidAngle();

    // Now compute the second coordinate using the new C vertex.

    Scalar z  = P | B() ;
    Scalar x2 = ( 1.0 - z ) / ( 1.0 - (C_new | B()) );

    u= Clamp( 0.0, x1, 1.0 ) ;
    v= Clamp( 0.0, x2, 1.0 ) ;
}

/*-------------------------------------------------------------------------*
 * D U A L                                                                 *
 *                                                                         *
 * Construct the dual triangle of the current triangle, which is another   *
 * spherical triangle.                                                     *
 *                                                                         *
 *-------------------------------------------------------------------------*/

SphericalTriangle 
Dual
( 
  const SphericalTriangle &T 
)

{
    Vector A = T.B() * T.C();  if( (A | T.A()) < 0.0 ) A *= -1.0;
    Vector B = T.A() * T.C();  if( (B | T.B()) < 0.0 ) B *= -1.0;
    Vector C = T.A() * T.B();  if( (C | T.C()) < 0.0 ) C *= -1.0;

    return SphericalTriangle( A, B, C );
}



