#include <math.h>

#include "Vector.H"
#include "Aux.H"

#include "ETriangle.H"
#include "PlanarObj.H"

//*******************************************

PlanarObject:: PlanarObject
(
   const Vector & iVector0 ,
   const Vector & iVector1 ,
   const Vector & iVector2
)  
{
   vertex[0] = iVector0 ;
   vertex[1] = iVector1 ;
   vertex[2] = iVector2 ;

   Vector edgeU = vertex[1] - vertex[0]  ,
          edgeV = vertex[2] - vertex[0]  ;

   // compute dual basis by substracting 
   // to each vector its projection onto the other, 
   //
   // after this we get : 
   //   'dualBaseU' perpendicular to 'edgeV', and
   //   'dualBaseV' perpendicular to 'edgeU'

   dualBaseU = edgeU - ( edgeV * ( edgeU | edgeV.Norm2() ) ) ;
   dualBaseV = edgeV - ( edgeU * ( edgeV | edgeU.Norm2() ) ) ;

   // scale the dual basis:
   //
   // after this we get: 
   //   inner product of 'dualBaseU' and 'edgeU' equal to 1, and
   //   inner product of 'dualBaseV' and 'edgeV' equal to 1
             
   dualBaseU *= 1.0 / (dualBaseU|edgeU) ;
   dualBaseV *= 1.0 / (dualBaseV|edgeV) ;

   // compute normal to the plane and
   // the 'k' constant (used in the intersection test).

   normal = (edgeU * edgeV).Normalized() ;

   k =  vertex[0] | normal  ;

   //  make 'tangentU' and 'tangentV' two orthonormal
   //  vectors on the plane (an alternative auto-dual basis).

   tangentU = edgeU.Normalized() ;
   tangentV = dualBaseV.Normalized() ;
}

//*****************************************
   
void PlanarObject::IntersectionTest
( 
   const Ray &  ray         ,
   Boolean &    hitFound    ,
   Scalar &     hitDistance ,
   Vector &     hitPoint
)  const 
{

   Scalar d = ray.direction | normal ;

   if ( d == 0 )
   {
      hitFound = FALSE ;
      return ;
   }   

   hitDistance = ( k - ( ray.origin | normal ) ) / d ;

   if ( hitDistance <= 1e-6 )
   {
      hitFound = FALSE ;
      return ;
   }

   hitFound = TRUE ;
   hitPoint = ray.origin + hitDistance * ray.direction ;
}

//*******************************************

Boolean PlanarObject::OccludesSegment
(
   const Vector & p1 ,
   const Vector & p2
)  const 
{
   Ray ray( p1, p2-p1 ) ;

   Boolean hitFound    ;
   Scalar  hitDistance ;
   Vector  hitPoint    ;

   IntersectionTest( ray, hitFound, hitDistance, hitPoint ) ;

   if ( ! hitFound )
      return FALSE ;

   if ( hitDistance < 1e-5 || 1.0-1e-5 < hitDistance )
      return FALSE ;

   return TRUE ;
}



//******************************************

LocalRefSys PlanarObject::GetLRS
(
   const Vector & origin 
)  const
{
   return LocalRefSys
          (  origin,
             normal,
             tangentU ,
             tangentV
          ) ;

}

//******************************************

Vector PlanarObject::GetNormal
(
)  const
{
   return normal ;
}


//****************************************

Triangle::Triangle
(
   const Vector & iVector0 ,
   const Vector & iVector1 ,
   const Vector & iVector2
)  
:  PlanarObject( iVector0, iVector1, iVector2 )
  
{

}

//*****************************************

void Triangle::IntersectionTest
( 
   const Ray & ray         ,
   Boolean &   hitFound    ,
   Scalar &    hitDistance ,
   Vector &    hitPoint
)  const 
{

   PlanarObject::IntersectionTest( ray, hitFound, hitDistance, hitPoint ) ;

   if ( ! hitFound )
      return ;

   Vector transHitPoint = hitPoint - vertex[0] ;

   Scalar u = transHitPoint | dualBaseU  ;
   Scalar v = transHitPoint | dualBaseV  ;

   hitFound =  (0.0 <= u)  &&  (0.0 <= v)  &&  ((u+v) <= 1.0)  ;
      
} 

//*******************************************

Vector Triangle::GetVertex
( 
   unsigned i 
)  const 
{
   return vertex[i] ;
} 

//******************************************

Scalar Triangle::GetFormFactor
(
   const LocalRefSys & lrs
)  const 
{

   ETriangle et(  lrs.WorldToLocal( vertex[0] ) ,
                  lrs.WorldToLocal( vertex[1] ) ,
                  lrs.WorldToLocal( vertex[2] ) ,
                  0.01
               )  ;
   
   return et.GetFormFactor() ;
}

//******************************************


Parallelogram::Parallelogram
(
   const Vector & iVector0 ,
   const Vector & iVector1 ,
   const Vector & iVector2
)  
:  PlanarObject( iVector0, iVector1, iVector2 )
{


}

//*****************************************

void Parallelogram::IntersectionTest
( 
   const Ray & ray         ,
   Boolean &   hitFound    ,
   Scalar &    hitDistance ,
   Vector &    hitPoint    
)  const 
{
   PlanarObject::IntersectionTest( ray, hitFound, hitDistance, hitPoint ) ;

   if ( ! hitFound )
      return ;

   Vector transHitPoint = hitPoint - vertex[0] ;

   Scalar u =  transHitPoint | dualBaseU  ;
   Scalar v =  transHitPoint | dualBaseV  ;
   
   hitFound =  (0.0 <= u)  &&  (0.0 <= v)  &&  (u <= 1.0) && (v <= 1.0) ;

} 

//*****************************************

RectangleSet::RectangleSet
(
   const Vector & iVector0 ,
   const Vector & iVector1 ,
   const Vector & iVector2 ,
   unsigned       inu      ,
   unsigned       inv
)  
:  PlanarObject( iVector0, iVector1, iVector2 ) ,
   nu( inu ) ,
   nv( inv )
{

}

//***************************************

void RectangleSet::IntersectionTest
( 
   const Ray & ray         ,
   Boolean &   hitFound    ,
   Scalar &    hitDistance ,
   Vector &    hitPoint
)  const 
{
   
   PlanarObject::IntersectionTest( ray, hitFound, hitDistance, hitPoint ) ;

   if ( ! hitFound )
      return ;

   Vector transHitPoint = hitPoint - vertex[0] ;

   Scalar u = transHitPoint | dualBaseU  ;
   Scalar v = transHitPoint | dualBaseV  ;

   if ( (u < 0.0) ||  (v < 0.0)  ||  (1.0 < u) || (1.0 < v) )
   {
      hitFound = FALSE ;
      return ;
   }
/**
   else
   {
      hitFound = TRUE ;
      return ;
   }
**/

   Scalar fu =  ((Scalar)nu) * u  ,
          fv =  ((Scalar)nv) * v  ;

   Scalar du = fabs( fu - floor( fu ) - 0.5 ) ;
   Scalar dv = fabs( fv - floor( fv ) - 0.5 ) ;

   Scalar m = ( du > dv ) ? du : dv ;

   if ( m < 0.25 ) 
      hitFound= TRUE ;
   else
      hitFound= FALSE ;
} 
