#include <math.h>

#include "lsix.h"
#include "Demo2.H"
#include "SamplingProc.H"


Demo2::Demo2
(
   Boolean   iMcGather ,
   unsigned  iNSamples ,
   Scalar    iEpsilon  
)
:  mcGather( iMcGather ) ,
   nSamples( iNSamples  ) ,
   epsilon(  iEpsilon ) 
{
   // camera parameters: 

   Vector lookAt  ( 0.0, 0.0, 0.0 ) ,
          obsPos  ( 2.0, 2.0, 2.0 ) ,
          vup     ( 0.0, 1.0, 0.0 ) ;

   Scalar horFovDegrees = 60.00 ;

   //
   // computation of camera reference system

   Vector vpn           = obsPos - lookAt ;
   Scalar horFovRadians = (horFovDegrees * M_PI) / 180.0 ;
   Scalar sinHFov       = sin( 0.5 * horFovRadians ) ;

   camU = sinHFov * ((vup * vpn ).Normalized())  ;
   camV = sinHFov * ((vpn * camU).Normalized()) ;
   camN = vpn.Normalized() ;

   camO = obsPos - camN ;
   camF = obsPos ;

   //
   // create objects

   source =
       new Triangle
           ( Vector( -0.001, 0.001, 0.00 ),
             Vector( -0.001, 1.500, 0.50 ),
             Vector( -0.001, 0.001, 1.00 )
           ) ;

   objs[iSource] = source ;

#define DY 0.0

   occluder =   
       new RectangleSet
           ( Vector( 0.1, DY + 0.03, 0.1 ) ,
             Vector( 0.1, DY + 0.03, 0.9 ) ,
             Vector( 0.8,  DY + 0.4, 0.1 ) ,
             3, 3
           )  ;

   objs[iOccluder] = occluder ;

   
   objs[iReceiver] = 
       new Parallelogram
           ( Vector( 0.0, 0.0, 0.0 ),
             Vector( 0.0, 0.0, 1.0 ),
             Vector( 1.0, 0.0, 0.0 )
           ) ;
   
   for( int i= 0 ; i < 3 ; i++ )
      if ( objs[i] == NULL )
         throw "Not enough memory for 'new'" ;
   

}

//***************************************************

Ray Demo2::GenRay
( 
   Scalar  u,
   Scalar  v
)  const 
{
   u = 2.0 * (u-0.5) ;
   v = 2.0 * (v-0.5) ;

   Vector viewPlanePoint = camO + (u*camU) + (v*camV) ;

   return  Ray( camF, (viewPlanePoint - camF).Normalized() );
}

//********************************************

Scalar Demo2::EvalSourceRadiance
(
   const Vector & sourcePoint
)  const 
{
   unsigned i = (unsigned)floor( sourcePoint[Z] * 5.0 ) ,
            j = (unsigned)floor( sourcePoint[Y] * 4.0 ) ;
 
   Scalar res ;

   res =  ( i & 1 ) ? 0.5 : 1.0 ;
   res *= ( j & 1 ) ? 1.0 : 0.5 ;

   return res * 1.7 ;
}

//********************************************


Scalar Demo2::GetIntensityAt
(
  Scalar  u ,
  Scalar  v
) const
{  

   Ray ray = GenRay( u, v ) ;

   Boolean hit    = FALSE ,
           first  = TRUE  ;

   Scalar  minDistance  ,
           res               ; 
   int     iObjHit           ;  
   Vector  point             ;

   for( unsigned iObj= 0 ; iObj < 3 ; iObj++ )
   {
      Boolean hitFound     ;
      Scalar  hitDistance  ;
      Vector  hitPoint     ;

      objs[iObj]->IntersectionTest( ray, hitFound, hitDistance, hitPoint ) ;

      if ( hitFound )
      {
         hit= TRUE ;

         if ( first || hitDistance < minDistance )
         {  
             iObjHit = iObj ;
             minDistance = hitDistance ;
             first = FALSE ;
             point = hitPoint ;
         }
      }
   }

   if ( ! hit )
      return 0.15 ;

   switch ( iObjHit )
   {
      case iSource    : 
     
          return EvalSourceRadiance( point ) ;

      case iReceiver  : 
      case iOccluder  :

          if ( mcGather )
             res =  Gather( point, iObjHit ) ;
          else
             res =  ExactGather( point, iObjHit ) ;

          return res ;

      default         : 
          throw "Invalid value for 'iObjHit' " ;
   }

}

//************************************************

Scalar Demo2::Gather
(
   const Vector & target ,
   unsigned       iObj 
)  const
{
   LocalRefSys lrs = objs[iObj]->GetLRS( target ) ;
   Vector      vertex[3] ;

   for( unsigned i= 0 ; i < 3 ; i++ )
      vertex[i] = source->GetVertex( i ) ;

   PsaSamplingProc psa( lrs, vertex[0], vertex[1], vertex[2], epsilon ) ; 

   Sample * sampleSet = new Sample[nSamples] ;

   if ( sampleSet == NULL )
      throw "Unable to alloc -- I need more memory !" ;

   psa.GetSamples( nSamples, sampleSet ) ;

   Scalar result= 0.0 ;

   Vector srcNor = source->GetNormal()   , 
          recNor = objs[iObj]->GetNormal() ;

   for( unsigned i = 0 ; i < nSamples ; i++ )
   {
       if ( occluder->OccludesSegment( target, sampleSet[i].position ) )
          continue ;
   
       Scalar Li = GetIrradiance( sampleSet[i].position, srcNor, target ) ,
              pu = sampleSet[i].probability ;

       result += Li/pu * (1.0/M_PI) ; // last factor is the diffuse BRDF with 
   }                                  // \rho == 1 (reflectivity or albedo)

   result /= ((Scalar) nSamples) ;

   delete [] sampleSet ;

   return result ;

}

//************************************************

Scalar Demo2::ExactGather
(
   const Vector & target ,
   unsigned       iObj 
)  const
{
   LocalRefSys lrs = objs[iObj]->GetLRS( target ) ;
   Vector      vertex[3] ;

   for( unsigned i= 0 ; i < 3 ; i++ )
      vertex[i] = lrs.WorldToLocal( source->GetVertex( i ) ).Normalized() ;

   ETriangle et( vertex[0], vertex[1], vertex[2], epsilon ) ; 

   Scalar result = et.GetFormFactor() / M_PI ;

   return result ;
}


//***********************************************

Scalar Demo2::GetIrradiance
(  
   const Vector & sample ,
   const Vector & sampleNor ,
   const Vector & target 
)  const 
{
   return EvalSourceRadiance( sample ) ;
}

//************************************************

void Demo2::DisplayPerspectiveView
(  
   unsigned resX ,
   unsigned resY
)  const 
{
   DWindow win("Demo2: 3d Scene",resX,resY) ;

   for( unsigned iRow = 0 ; iRow < resY ; iRow++ )
   {
      for( unsigned iCol = 0 ; iCol < resX ; iCol++ )
      {
         Scalar u = (Scalar) iCol / (Scalar) resX ,
                v = (Scalar) iRow / (Scalar) resY ;

         Scalar intensity = GetIntensityAt( u , v ) ;

         win.SetPixelAt( iCol, resY - iRow,  Correct(intensity) ) ;

      }
  //    cout << "\x0D " << ((Scalar)iRow*100.0/(Scalar)resX) << "% completed" << flush ;
   }

   Scalar dummy ; 
//   cout << "\x0D 100% completed                " << endl << flush ;

   win.GetPoint( dummy, dummy ) ;
}

//********************************

void Demo2::DisplayTopView
(  
   unsigned resX ,
   unsigned resY
)  const 
{
   DWindow win("Demo2: 3d Scene - top view",resX,resY) ;

   unsigned ix, iy ;

   for( ix = 0 ; ix < resX ; ix++ )
   for( iy = 0 ; iy < resY ; iy++ )
   {
      Scalar pz = 1.0 - (Scalar) ix / (Scalar) resX ,
             px = (Scalar) iy / (Scalar) resX ; 
      
      Vector recPoint( px, 0.0, pz ) ;

      Scalar b = Gather( recPoint, iReceiver ) ;      
      win.SetPixelAt( ix, iy, Correct(b) ) ;
   }

   Scalar dummy ;

   win.GetPoint( dummy, dummy ) ;
}

//********************************

Scalar Demo2::Correct
(
   Scalar b 
)  const 
{
   return b >= 1.0 ? 0.99 : b ;
}

//********************************

void Demo2::Run
(
   unsigned topView
)
{
   try
   {
      if ( topView )
         DisplayTopView( 200, 100 ) ;
      else 
         DisplayPerspectiveView( 500, 500 ) ;
   }
   catch( char * msg )
   {
      cerr << endl << "error raised, display aborted, msg follows:"
           << endl << msg 
           << endl << flush ;

      exit(1) ;
   }
}


