#include <ModeloGRF.H>
#include <ParserError.H>
#include <values.h>
#include <math.h>
#include <Chrono.H>

#include <GL/gl.h>


namespace GRF
{


ModeloGRF::ModeloGRF
(
  const char * fileName
)
: Modelo3D()
, parser(NULL)
, maxColorComputed( false )
{
  parser = new Parser(fileName) ;

  if ( parser == NULL )
     throw ParserError("cannot create Parser","ModeloGRF::ModeloGRF") ;

  unsigned long iv, il ;

  bBox.x0 = bBox.y0 = bBox.z0 =  MAXDOUBLE ;
  bBox.x1 = bBox.y1 = bBox.z1 =  MINDOUBLE ;

  double p[3] ;

  for( iv=0 ; iv < parser->NVertexs() ; iv++ )
  {
     parser->GetVertex(iv)->GetPosition(p) ;

     if ( p[0] < bBox.x0 ) bBox.x0 = p[0] ;
     if ( p[1] < bBox.y0 ) bBox.y0 = p[1] ;
     if ( p[2] < bBox.z0 ) bBox.z0 = p[2] ;

     if ( p[0] > bBox.x1 ) bBox.x1 = p[0] ;
     if ( p[1] > bBox.y1 ) bBox.y1 = p[1] ;
     if ( p[2] > bBox.z1 ) bBox.z1 = p[2] ;
  }

  double v[3][3], origin[3], edge0[3], edge1[3] ;

  for( il=0 ; il < parser->NLights() ; il++ )
  {
     const Light * l = parser->GetLight(il) ;

     l->GetOrigin( origin ) ;
     l->GetEdge0( edge0 ) ;
     l->GetEdge1( edge1 ) ;

     for( int i=0 ; i< 3 ; i++ )
     {
        v[i][0] = origin[i] ;
        v[i][1] = origin[i] + edge0[i] ;
        v[i][2] = origin[i] + edge1[i] ;
     }

     for( int i=0 ; i < 3 ; i++ )
     {
        if ( v[0][i] < bBox.x0 ) bBox.x0 = v[0][i] ;
        if ( v[1][i] < bBox.y0 ) bBox.y0 = v[1][i] ;
        if ( v[2][i] < bBox.z0 ) bBox.z0 = v[2][i] ;

        if ( v[0][i] > bBox.x1 ) bBox.x1 = v[0][i] ;
        if ( v[1][i] > bBox.y1 ) bBox.y1 = v[1][i] ;
        if ( v[2][i] > bBox.z1 ) bBox.z1 = v[2][i] ;
     }
  }



}



void ModeloGRF::KeyPressEvent
(
   unsigned char ch
)
{
   switch( ch )
   {
       case 'p' : case 'P' :
          params.renderStyle = rsPoints ;
          break ;

       case 'w' : case 'W' :
          params.renderStyle = rsWireframe ;
          break ;

       case 'f' : case 'F' :
          params.renderStyle = rsSolidFlatShaded ;
          std::cout << std::endl << "solid, flat shaded rendering" << std::flush ;
          break ;

       case 's' : case 'S' :
          params.renderStyle = rsSolidSmoothShaded ;
          std::cout << std::endl << "solid, smooth shaded rendering" << std::flush ;
          break ;

       case 'i' : case 'I' :
          params.renderStyle = rsIradiance ;
          std::cout << std::endl << "rendering face color * vertexs iradiance" << std::flush ;
          break ;

       case 'n' : case 'N' :
          params.reverseNormals = ! params.reverseNormals ;
          if ( params.reverseNormals )
             std::cout << std::endl << "reversing normals." << std::flush ;
          else
             std::cout << std::endl << "not reversing normals, keeping originals" << std::flush ;
          break ;

       case 'o' : case 'O' :
          params.reverseFacesOrientation = ! params.reverseFacesOrientation ;
          if ( params.reverseFacesOrientation )
             std::cout << std::endl << "reversing faces orientation." << std::flush ;
          else
             std::cout << std::endl << "not reversing faces orientation, keeping original ordering" << std::flush ;
          break ;

       case 'd' : case 'D' :
          params.drawNormals = ! params.drawNormals ;
          if ( params.drawNormals )
             std::cout << std::endl << "drawing normals" << std::flush ;
          else
             std::cout << std::endl << "not drawing normals" << std::flush ;
          break ;

       case 'G' :
          params.gamma += 0.1 ;
          std::cout << std::endl << "gamma increased to " << params.gamma << std::flush ;
          break ;

       case 'g' :
          if ( params.gamma > 1.0 )
          {
             params.gamma -= 0.1 ;
             std::cout << std::endl << "gamma decreased to " << params.gamma << std::flush ;
          }
          else
             std::cout << std::endl << "cannot decrease gamma, already equal to 1.0 (minimun)" << std::flush ;
          break ;
   }

   glutPostRedisplay() ;
}

Caja ModeloGRF::LeerCajaEnglobante()
{
  return bBox ;
}

//********************************************************************

bool lighting = true ;

GLfloat redColor[] = { 1.0, 0.0, 0.0, 1.0 } ;
GLfloat red8Color[] = { 0.8, 0.0, 0.0, 1.0 } ;
GLfloat red2Color[] = { 0.2, 0.0, 0.0, 1.0 } ;

GLfloat cyan5Color[] = { 0.0, 0.5, 0.5, 1.0 } ;
GLfloat cyan2Color[] = { 0.0, 0.2, 0.2, 1.0 } ;

GLfloat whiteColor[] = { 1.0, 1.0, 1.0, 1.0 } ;
GLfloat grayColor[] = { 0.5, 0.5, 0.5, 0.5 } ;
GLfloat gray2Color[] = { 0.2, 0.2, 0.2, 0.2 } ;

GLfloat blackColor[] = { 0.0 ,0.0, 0.0 ,0.0 } ;



void ModeloGRF::SetLights()
{

    glLightModeli( GL_LIGHT_MODEL_TWO_SIDE, 1 ) ;

    glEnable(GL_LIGHTING) ;
    glEnable(GL_LIGHT0) ;
    glEnable(GL_LIGHT1) ;
    

    GLfloat pos0[] = {  1.0,  1.0,  1.0 , 0.0 } ;
    GLfloat pos1[] = { -1.0, -1.0, -1.0,  0.0 } ;
    GLfloat pos2[] = {  0.0,  1.0,  1.0,  0.0 } ;

    glLightfv(GL_LIGHT0, GL_POSITION, pos0 ) ;
    glLightfv(GL_LIGHT0, GL_DIFFUSE,  whiteColor ) ;
    glLightfv(GL_LIGHT0, GL_SPECULAR, whiteColor ) ;
    glLightfv(GL_LIGHT0, GL_AMBIENT,  whiteColor ) ;

    glLightfv(GL_LIGHT1, GL_POSITION, pos1 ) ;
    glLightfv(GL_LIGHT1, GL_DIFFUSE,  grayColor ) ;
    glLightfv(GL_LIGHT1, GL_SPECULAR, blackColor ) ;
    glLightfv(GL_LIGHT1, GL_AMBIENT,  blackColor ) ;

    glLightfv(GL_LIGHT2, GL_POSITION, pos2 ) ;
    glLightfv(GL_LIGHT2, GL_DIFFUSE,  blackColor ) ;
    glLightfv(GL_LIGHT2, GL_SPECULAR, whiteColor ) ;
    glLightfv(GL_LIGHT2, GL_AMBIENT,  blackColor ) ;

    //std::cout << "ModeloGRF::SetLights" << std::endl << std::flush ;
}

//********************************************************************

#define rnd() ((double)random()/(double)RAND_MAX)


void ModeloGRF::GetFaceNormal
(
   const Face * f ,
   double n[3]
)
{
   double v0[3],v1[3],v2[3], a1[3], a2[3] ;


   parser->GetVertex( f->GetVertexIndex(0) )->GetPosition(v0) ;
   parser->GetVertex( f->GetVertexIndex(1) )->GetPosition(v1) ;
   parser->GetVertex( f->GetVertexIndex(2) )->GetPosition(v2) ;

   a1[0] = v1[0] - v0[0] ;
   a1[1] = v1[1] - v0[1] ;
   a1[2] = v1[2] - v0[2] ;

   a2[0] = v2[0] - v0[0] ;
   a2[1] = v2[1] - v0[1] ;
   a2[2] = v2[2] - v0[2] ;

   n[0] = a1[1]*a2[2] - a2[1]*a1[2] ;
   n[1] = a2[0]*a1[2] - a1[0]*a2[2] ;
   n[2] = a1[0]*a2[1] - a2[0]*a1[1] ;

   double l = sqrt( n[0]*n[0] + n[1]*n[1] + n[2]*n[2] ) ;

   n[0] /= l ;
   n[1] /= l ;
   n[2] /= l ;

   if ( params.reverseNormals )
   {
      n[0] *= -1.0 ;
      n[1] *= -1.0 ;
      n[2] *= -1.0 ;
   }
}

//********************************************************************


void ModeloGRF::DrawPoints()
{
    const Vertex *  v ;
    const Face *    f ;
    double          p[3] ;

    glPolygonMode( GL_FRONT_AND_BACK, GL_POINT ) ;
    glDisable( GL_LIGHTING ) ;
    glDisable( GL_TEXTURE_2D ) ;


    for( unsigned long i = 0 ; i < parser->NFaces() ; i++ )
    {
       f = parser->GetFace(i) ;

       glBegin(GL_POLYGON);

       for( unsigned long j= 0 ; j < f->NVertexs() ; j++ )
       {
          v = parser->GetVertex( f->GetVertexIndex(j) ) ;

          v->GetPosition( p ) ;
          glVertex3dv( p ) ;

       }
       glEnd();
    }
}

//********************************************************************

void ModeloGRF::DrawWireframe()
{

    const Vertex *       v ;
    const Face *         f ;
    unsigned int         vindex ;

    double p[3] ;

    glPolygonMode( GL_FRONT_AND_BACK, GL_LINE ) ;
    glDisable( GL_LIGHTING ) ;
    glDisable( GL_TEXTURE_2D ) ;

    for( unsigned long i = 0 ; i < parser->NFaces() ; i++ )
    {
       f = parser->GetFace(i) ;

       glBegin(GL_POLYGON);

       for( unsigned long j=0 ; j < f->NVertexs() ; j++ )
       {

          if ( params.reverseFacesOrientation )
             vindex = f->NVertexs()-j-1 ;
          else
             vindex = j ;

          v = parser->GetVertex( f->GetVertexIndex(vindex) ) ;


          v->GetPosition( p ) ;
          glVertex3dv( p ) ;

       }
       glEnd();
    }


}

//********************************************************************

inline void SetMaterialColor
(
   const Face * f
)
{
   float color8[4], color2[4] ;
   double  faceColor[3] ;

   f->GetColor( faceColor ) ;

   for( int i=0 ; i < 3 ; i++ )
   {
      color8[i] = faceColor[i] * 0.8 ;
      color2[i] = faceColor[i] * 0.2 ;
   }

   color8[3] = 1.0 ;
   color2[3] = 1.0 ;

   glMaterialfv( GL_FRONT, GL_DIFFUSE,  color8 ) ;
   glMaterialfv( GL_FRONT, GL_AMBIENT,  color2 ) ;

   glDisable(GL_TEXTURE_2D);
}

//********************************************************************

inline const GRF::Texture * ActivateMaterialTexture
(
   Parser * parser ,
   long     prevFaceTexture ,
   long     faceTexture
)
{
   float white8[4] = { 0.8, 0.8, 0.8, 1.0 } ;
   float white2[4] = { 0.2, 0.2, 0.2, 1.0 } ;

   glMaterialfv( GL_FRONT, GL_DIFFUSE,  white8 ) ;
   glMaterialfv( GL_FRONT, GL_AMBIENT,  white2 ) ;

   const GRF::Texture * tex = parser->GetTexture( faceTexture ) ;

   if ( prevFaceTexture != faceTexture )
      tex->GL_ActivateTexture() ;

   return tex ;
}

//********************************************************************

const Vertex * ModeloGRF::GetVertex
(
   const Face * f,
   int          iv
)
{
    int index ;

    if ( params.reverseFacesOrientation )
       index = f->NVertexs()-iv-1 ;
    else
       index = iv ;

    return parser->GetVertex( f->GetVertexIndex(index) ) ;

}

//********************************************************************

void ModeloGRF::DrawSolidFlatShaded()
{

    const Vertex *       v ;
    const Face *         f ;
    int                  prevFaceTexture = -1 ;
    const GRF::Texture * tex = NULL ;

    double p[3], fn[3]  ;
    double uv[2] ;

    glPolygonMode( GL_FRONT_AND_BACK, GL_FILL ) ;
    glEnable( GL_LIGHTING ) ;
    glShadeModel( GL_FLAT ) ;


    GLfloat brillo[] = { 50.0 } ;

    glMaterialfv( GL_FRONT, GL_SHININESS, brillo ) ;
    glMaterialfv( GL_FRONT, GL_SPECULAR, whiteColor ) ;

    glMaterialfv( GL_BACK, GL_SPECULAR, blackColor ) ;
    glMaterialfv( GL_BACK, GL_DIFFUSE,  cyan5Color ) ;
    glMaterialfv( GL_BACK, GL_AMBIENT,  cyan2Color ) ;


    for( unsigned long i = 0 ; i < parser->NFaces() ; i++ )
    {
       f = parser->GetFace(i) ;


       long faceTexture = f->GetTextureNumber() ;

       if ( faceTexture == -1 )
       {
          SetMaterialColor(f);
          prevFaceTexture = -1 ;
          tex = NULL ;
       }
       else
       {
          tex = ActivateMaterialTexture( parser, prevFaceTexture, faceTexture ) ;
          prevFaceTexture = faceTexture ;
       }

       GetFaceNormal(f,fn) ;

       glBegin(GL_POLYGON);

       for( unsigned long j=0 ; j < f->NVertexs() ; j++ )
       {

          v = GetVertex(f,j);

          glNormal3dv( fn ) ;

          if ( tex != NULL )
          if ( v->HasTextCoords() )
          {
             v->GetTextCoords(uv) ;
             tex->GL_SetTextureCoordinates(uv[0],uv[1]);
          }

          v->GetPosition( p ) ;
          glVertex3dv( p ) ;

       }
       glEnd();
    }


}

//******************************************************************

void ModeloGRF::DrawSolidSmoothShaded()
{
    const Vertex *       v ;
    const Face *         f ;

    int                  prevFaceTexture = -1 ;
    const GRF::Texture * tex = NULL ;

    double p[3], n[3] ;
    double uv[2] ;

    glPolygonMode( GL_FRONT_AND_BACK, GL_FILL ) ;
    glEnable( GL_LIGHTING ) ;
    glShadeModel( GL_SMOOTH ) ;

    GLfloat brillo[] = { 50.0 } ;

    glMaterialfv( GL_FRONT, GL_SHININESS, brillo ) ;
    glMaterialfv( GL_FRONT, GL_SPECULAR, whiteColor ) ;

    glMaterialfv( GL_BACK, GL_SPECULAR, blackColor ) ;
    glMaterialfv( GL_BACK, GL_DIFFUSE,  cyan5Color ) ;
    glMaterialfv( GL_BACK, GL_AMBIENT,  cyan2Color ) ;



    for( unsigned long i = 0 ; i < parser->NFaces() ; i++ )
    {
       f = parser->GetFace(i) ;

       long faceTexture = f->GetTextureNumber() ;

       if ( faceTexture == -1 )
       {
          SetMaterialColor(f);
          prevFaceTexture = -1 ;
          tex = NULL ;
       }
       else
       {
          tex = ActivateMaterialTexture( parser, prevFaceTexture, faceTexture ) ;
          prevFaceTexture = faceTexture ;
       }


       glBegin(GL_POLYGON);


       for( unsigned long j=0 ; j < f->NVertexs() ; j++ )
       {

          v = GetVertex(f,j);

          if (  v->HasNormal() )
          {
             v->GetNormal( n ) ;
             if ( params.reverseNormals )
             {
                n[0] = -n[0] ;
                n[1] = -n[1] ;
                n[2] = -n[2] ;
             }

             glNormal3dv(n);
          }

          if ( tex != NULL )
          if ( v->HasTextCoords() )
          {
             v->GetTextCoords(uv) ;
             tex->GL_SetTextureCoordinates(uv[0],uv[1]);
          }

          v->GetPosition( p ) ;
          glVertex3dv( p ) ;

       }
       glEnd();
    }

}

void ModeloGRF::DrawIradiance()
{
    const Vertex * v ;
    const Face *   f ;
    int            prevFaceTexture = -1 ;
    const GRF::Texture *
                   tex = NULL ;
    double         vertexIradiance[3],
                   vertexColor[3] ,
                   p[3] ,
                   colorFactor = 1.0 ,
                   m = GetMaxColor() ;


    glPolygonMode( GL_FRONT , GL_FILL ) ;
    glPolygonMode( GL_BACK,  GL_LINE ) ;
    glDisable( GL_LIGHTING ) ;
    glShadeModel( GL_SMOOTH ) ;



    if ( m > 0.0 )
       colorFactor = (double)1.0/m ;
    else
       std::cout << "warning: max color is 0.0" ;


    for( unsigned long i = 0 ; i < parser->NFaces() ; i++ )
    {
       f = parser->GetFace(i) ;

       double faceColor[3] ;
       long faceTexture = f->GetTextureNumber() ;

       f->GetColor(faceColor);


       if ( faceTexture == -1 )
       {
          glDisable( GL_TEXTURE_2D ) ;
          tex = NULL ;
       }
       else
       {
          tex = parser->GetTexture( faceTexture ) ;

          if ( prevFaceTexture != faceTexture )
             tex->GL_ActivateTexture() ;

       }

       prevFaceTexture = faceTexture ;

       glBegin(GL_POLYGON);

       double invGamma = 1.0 ;

       if ( params.gamma != 0.0 )
          invGamma = 1.0/params.gamma ;

       for( unsigned long j=0 ; j < f->NVertexs() ; j++ )
       {
          v = GetVertex(f,j);

          v->GetIradiance(vertexIradiance) ;


          if ( tex == NULL )
          {
             for( int i=0 ; i< 3 ; i++ )
             {
                vertexColor[i] = colorFactor * vertexIradiance[i] * faceColor[i] ;

                if ( params.gamma != 1.0 )
                   vertexColor[i] = pow(vertexColor[i],invGamma) ;
             }
          }
          else
          {
             for( int i=0 ; i< 3 ; i++ )
             {
                vertexColor[i] = colorFactor * vertexIradiance[i] ;

                if ( params.gamma != 1.0 )
                   vertexColor[i] = pow(vertexColor[i],invGamma);
             }
          }


          glColor3dv( vertexColor ) ;

          if ( tex != NULL )
          if ( v->HasTextCoords() )
          {
             double uv[2] ;

             v->GetTextCoords(uv) ;
             tex->GL_SetTextureCoordinates(uv[0],uv[1]);
          }


          v->GetPosition( p ) ;
          glVertex3dv( p ) ;

       }
       glEnd();
    }

}

//*******************************************************************

void ModeloGRF::DrawModelFaces()
{

    switch ( params.renderStyle )
    {
        case rsPoints :

           DrawPoints() ;
           break ;

        case rsWireframe :

           DrawWireframe() ;
           break ;

        case rsSolidFlatShaded :

           DrawSolidFlatShaded() ;
           break ;

        case rsSolidSmoothShaded :

           DrawSolidSmoothShaded() ;
           break ;

        case rsIradiance :

           DrawIradiance() ;
           break ;
    }
}

//********************************************************************


void ModeloGRF::Dibujar()
{
    Chrono chrono ;

    //std::cout << std::endl << "drawing ....." << std::flush ;

    chrono.Run() ;

    DrawModelFaces() ;
    DrawLightSources() ;

    if ( params.drawNormals )
       DrawNormals() ;

    chrono.Stop() ;

    //std::cout << "end, this took " << chrono.GetSeconds() << " seconds." << std::flush  ;
}

//********************************************************************

void ModeloGRF::DrawLightSources()
{
     glPushAttrib( GL_LIGHTING_BIT ) ;

     glDisable(GL_LIGHTING) ;
     glPolygonMode( GL_FRONT, GL_FILL ) ;
     glPolygonMode( GL_BACK, GL_LINE ) ;

     glColor3d( 1.0, 1.0, 1.0 ) ;

     // for each light source ....

     double v[3][4], origin[3], edge0[3], edge1[3], rgb[3] ;

     for( unsigned il=0 ; il < parser->NLights() ; il++ )
     {

        // get lights vertexs:

        const Light * l = parser->GetLight(il) ;

        l->GetOrigin( origin ) ;
        l->GetEdge0( edge0 ) ;
        l->GetEdge1( edge1 ) ;
        l->GetRGB( rgb ) ;

        for( int i= 0 ; i<3 ; i++ )
        {
           v[i][0] = origin[i] ;
           v[i][1] = origin[i] + edge0[i] ;
           v[i][2] = origin[i] + edge0[i] + edge1[i] ;
           v[i][3] = origin[i] + edge1[i] ;
        }

        // draw polygon:

        glColor3dv( rgb ) ;

        glBegin(GL_POLYGON) ;

          for( int i=0 ; i<4 ; i++ )
          {
             glVertex3d(v[0][i],v[1][i],v[2][i]) ;
          }

        glEnd() ;
     }

     glPopAttrib() ;
}

//********************************************************************

inline double cuad( double x ) { return x*x ; }

void ModeloGRF::DrawNormals()
{
  double l,diag ;

  glPushAttrib( GL_LIGHTING_BIT ) ;

  diag = sqrt( cuad( bBox.x1-bBox.x0 ) +  cuad( bBox.y1-bBox.y0 ) + cuad( bBox.z1-bBox.z0 ) ) ;

  l = diag / 30.0 ;

  glDisable( GL_LIGHTING ) ;
  glColor3d( 0.5, 1.0, 0.5 ) ;

  const Vertex * v ;
  double p[3], n[3], p2[3] ;

  for( unsigned long iv=0 ; iv < parser->NVertexs() ; iv++ )
  {

          v = parser->GetVertex( iv ) ;

          v->GetPosition( p ) ;
          v->GetNormal( n ) ;

          if ( params.reverseNormals )
          {
             n[0] = -n[0] ;
             n[1] = -n[1] ;
             n[2] = -n[2] ;
          }

          //std::cout  << std::endl << "nor+mal = " << n[0] << "," << n[1] << "," << n[2] << std::flush ;

          p2[0] = p[0] + l*n[0] ;
          p2[1] = p[1] + l*n[1] ;
          p2[2] = p[2] + l*n[2] ;

          glBegin( GL_LINES ) ;
             glVertex3d( p[0], p[1], p[2] ) ;
             glVertex3d( p2[0], p2[1], p2[2] ) ;
          glEnd( ) ;

  }

  glPopAttrib() ;

}

//********************************************************************

void ModeloGRF::CallBackDesocupado()
{
   static long unsigned int c = 0 ;

   c++ ;

   if ( c == 1000000 )
   {
       //std::cout << std::endl << "hola!" << std::flush ;
       c= 0 ;
   }

}

//********************************************************************

double ModeloGRF::GetMaxColor()
{
  if ( maxColorComputed )
     return maxColor ;

  maxColor = 0.0 ;

  unsigned long nFaces = parser->NFaces() ;

  for( unsigned long iFace = 0 ; iFace < nFaces ; iFace++ )
  {
     const Face * f = parser->GetFace( iFace ) ;
     const Texture * txt = NULL ;

     double frgb[3] ;

     long itext = f->GetTextureNumber() ;

     if ( itext == -1 )
        f->GetColor( frgb ) ;
     else
        txt = parser->GetTexture( itext ) ;


     unsigned long nvf = f->NVertexs() ;

     for( unsigned ivf =0 ; ivf < nvf ; ivf++ )
     {
        unsigned long iVertex = f->GetVertexIndex( ivf ) ;
        const Vertex * v = parser->GetVertex( iVertex ) ;

        double vrgb[3] ;

        v->GetIradiance( vrgb ) ;

        if ( txt != NULL )
        {
           double uv[2] ;

           v->GetTextCoords( uv ) ;
           txt->GetRGB( uv[0], uv[1], frgb ) ;
        }

        for( int i=0 ; i < 3 ; i++ )
        {
           double c = vrgb[i]*frgb[i] ;

           if ( c > maxColor )
              maxColor = c ;
        }
     }

  }


  maxColorComputed = true ;
  return maxColor ;
}


void ModeloGRF::GetModelInfo
(
   std::ostream & os
)
   const
{

   os << std::endl << "num. vertexs  : " << parser->NVertexs()
      << std::endl << "num. faces    : " << parser->NFaces()
      << std::endl << "num. lights   : " << parser->NLights()
      << std::endl << "num. textures : " << parser->NTextures()
      << std::endl << "bounding box: : (" << bBox.x0 << "," << bBox.y0 << "," << bBox.z0 << ")"
      << std::endl << "                (" << bBox.x1 << "," << bBox.y1 << "," << bBox.z1 << ")"
   ;

}

void ModeloGRF::SaveAs
(
   const char * fileName
)
{
   parser->SaveAs( fileName ) ;
}


} ; // namespace GRF

