//*************************************************************
//**
//**  File :     ImageWriter.C
//**  Function : Implementation of class "ImageWriter"
/* 2002-06-17:
 * Replaced all references to variables defined in namespace std, which
 * were used with no prefix, with std::<reference>
 * in order to compile with gcc 3.1.1, which no longer has 
 * "using namespace standard"
 * Rubn J. Garca Hernndez
 */ 
//**
//*************************************************************


#include <ImageWriter.H>
#include <ParserError.H>


namespace IL
{


using GRF::ParserError ;

inline byte low( unsigned int bb )
{
    return (bb & 0xFF) ;
}

inline byte high( unsigned int bb )
{
    return (bb >> 8) ;
}

ImageWriter::ImageWriter
(
   const std::string &  _fileName,
   unsigned        _sizeX,
   unsigned        _sizeY
)
:  fileName(_fileName)
,  sizeX(_sizeX)
,  sizeY(_sizeY)
,  data(NULL)
{

   dataSize = (unsigned long)sizeX * (unsigned long)sizeY * (unsigned long)3;

   data = new byte [dataSize] ;

   if ( data == NULL )
      throw ParserError("new failed, cannot alloc ","ImageWriter::ImageWriter");

    // general

    header.IDLength  = 0 ;    // length of string with additional info
    header.CoMapType = 0 ;    // color map type 0 --> no map
    header.ImgType   = 2 ;    // image type 2     --> uncompressed true color

    // color map (not used with this image type)

    header.Index_lo  = 0 ;
    header.Index_hi  = 0 ;     // index of first color map entry
    header.Length_lo = 0 ;
    header.Length_hi = 0 ;     // number of entries in color map
    header.CoSize    = 24 ;    // size of color map entry (15,16,24,32)

    // image

    header.X_org_lo   = 0 ;
    header.X_org_hi   = 0 ;            // x origin of image
    header.Y_org_lo   = 0 ;
    header.Y_org_hi   = 0 ;            // y origin of image
    header.Width_lo   = low(sizeX)  ;
    header.Width_hi   = high(sizeX) ;  // width of image
    header.Height_lo  = low(sizeY) ;
    header.Height_hi  = high(sizeY) ;  // height of image
    header.PixelSize  = 24 ;           // pixel size (8,16,24,32)
    header.Descriptor = 0 ;            // image descriptor (usually 0 for non-interlaced)

}

ImageWriter::~ImageWriter()
{
    if ( data != NULL )
       delete [] data ;
}


void ImageWriter::SetPixel
(
   unsigned ix,
   unsigned iy,
   byte     rgb[3]
)
{
   unsigned long offset = ((unsigned long)iy * (unsigned long)sizeX) + (unsigned long)ix ;


   byte * p = data + (offset*(unsigned long)3) ;

   *(p++) = rgb[2] ;  // blue
   *(p++) = rgb[1] ;  // green
   *(p++) = rgb[0] ;  // red
}

void ImageWriter::Write()
{
   FILE * file ;
   size_t nb ;

   file = fopen(fileName.c_str(),"w");

   if ( file == NULL )
   {
      std::string s = "unable to open file for writing image: " ;
      s += fileName ;

      throw ParserError(s,"ImageWriter::Write") ;
   }

   nb = fwrite( (void *)(& header), sizeof(TargaFileHeader), 1, file ) ;

   if ( nb != 1 )
      throw ParserError("cannot write image header, write failed","ImageWriter::Write") ;

   nb = fwrite( (void *)data, 1, dataSize, file ) ;

   if ( nb < dataSize )
      throw ParserError("unable to write targa image: couldn't complete writing data","ImageWriter::Write");

   fclose(file);
}

void ImageWriter::SwapRGB()
{
   for( unsigned long i = 0 ; i < dataSize ; i += 3 )
   {
      byte tmp  = data[i] ;

      data[i]   = data[i+2] ;
      data[i+2] = tmp ;
   }
}


} ; // namespace IL




