//*************************************************************
//**
//**  File :     ImageLoader.C
//**  Function : Implementation of namespace "ImageLoader"
/* 2002-06-17:
 * Replaced all references to variables defined in namespace std, which
 * were used with no prefix, with std::<reference>
 * in order to compile with gcc 3.1.1, which no longer has 
 * "using namespace standard"
 * Rubn J. Garca Hernndez
 */ 
//**
//*************************************************************



#include <ImageLoader.H>
#include <ParserError.H>
#include <unistd.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <errno.h>


namespace IL
{

using GRF::ParserError ;


ImageLoader::ImageLoader
(
   const std::string & targetFileName
)
{
   Init() ;
   LoadImage( targetFileName ) ;
}


ImageLoader::ImageLoader
()
{
   Init() ;
}

void ImageLoader::Init()
{
   file = NULL ;
   data = NULL ;

   imageLoaded = false ;

   fileName = "" ;

   xSize = 0 ;
   ySize = 0 ;
}

void ImageLoader::LoadImage
(
   const std::string & targetFileName
)
{
   //cout << endl << "loading image file " << targetFileName << " ..." << flush ;

   fileName = targetFileName ;

   CheckAndConvert() ;
   OpenFile() ;
   LoadTargaHeader() ;
   LoadData() ;

   imageLoaded = true ;

   //cout << endl << "image file " << targetFileName << " loaded ok." << flush ;
}

bool ImageLoader::ImageLoaded
()
   const
{
   return imageLoaded ;
}

unsigned int ImageLoader::XSize
()
   const
{
   return xSize ;
}

unsigned int ImageLoader::YSize
()
   const
{
   return ySize ;
}


void ImageLoader::GetPixel
(
   unsigned int x,
   unsigned int y,
   unsigned char pixel[3]
)
   const
{
   if ( ! imageLoaded )
      throw ParserError("internal error: cannot read pixel: targa image not loaded","ImageLoader::GetPixel");

   typedef unsigned long long T ;

   T offset = T(3) * ( T(y)*T(xSize) + T(x) ) ;

   byte * p = data + ((unsigned long)offset) ;

   pixel[2] = *(p++) ;
   pixel[1] = *(p++) ;
   pixel[0] = *(p++) ;

}

void ImageLoader::Close()
{

   //cout << endl << "closing image buffer ..." << flush ;

   if ( data != NULL )
      delete [] data ;


   if ( file != NULL )
      fclose(file) ;

   Init() ;

   //cout << " ok." << flush ;
}




ImageLoader::~ImageLoader()
{
   Close() ;
}

//****** private methods:

void ImageLoader::OpenFile
()
{
   file = fopen(fileName.c_str(),"r") ;

   if ( file == NULL )
   {
      std::string s = "unable to open file: " ;
      s += fileName ;

      throw ParserError(s,"ImageLoader::OpenFile") ;
   }
}

void ImageLoader::LoadTargaHeader()
{

#define THISF "ImageLoader::LoadTargaHeader"

   if ( file == NULL )
      throw ParserError("cannot load header, internal error: file not opened.",THISF) ;

   if ( feof(file) )
      throw ParserError("cannot load header, premature end of file",THISF);

   int nb = fread( (void *)(& header), sizeof(TargaFileHeader), 1, file ) ;

   if ( nb != 1 )
      throw ParserError("cannot load header, read failed",THISF) ;

   if (  ( header.CoMapType != 0 ||  header.ImgType != 2) )
      throw ParserError("file not readable: not a targa uncompressed 24 bit file",THISF) ;

   xSize = ( (unsigned int) header.Width_lo  + ((unsigned int)header.Width_hi  * 256) );
   ySize = ( (unsigned int) header.Height_lo + ((unsigned int)header.Height_hi * 256) );

   if ( xSize == 0 || ySize == 0 )
      throw ParserError("file not readable: x-size (or y-size) is cero",THISF);

   unsigned int idlength = (unsigned int)header.IDLength ;


   if ( idlength > 0 )
   {
       char str[256] ;

       unsigned int nb = fread( str, 1, idlength, file ) ;

       if ( nb < idlength )
           throw ParserError("cannot load identifier string, read failed",THISF);

       str[idlength] = (char)0 ;

       //cout << endl << "identifier string loaded"
       //     << endl << "[" << str  << "]" << flush ;
   }


}

void ImageLoader::LoadData
()
{

   //cout << endl << "loading image data ..." << flush ;

#define THISF1 "ImageLoader::LoadData"

   typedef unsigned long long st ;

   st dataSizeLong = st(3) * st(xSize) * st(ySize) ;

   size_t dataSize = dataSizeLong ;

   if ( st(dataSize) != dataSizeLong )
      throw ParserError("unable to read targa image: file too big",THISF1) ;

   data = new byte[dataSize] ;

   if ( data == NULL )
      throw ParserError("unable to read targa image: cannot alloc memory",THISF1);

   if ( feof(file) )
      throw ParserError("unable to read targa image: premature end of file just after header",THISF1);

   size_t nb = fread( (void *)data, 1, dataSize, file ) ;

   if ( nb < dataSize )
      throw ParserError("unable to read targa image: data size too short",THISF1);

   fclose(file);
   file = NULL ;

   //cout << " ok." << flush ;

}

void ImageLoader::ExecConvert
(
   const std::string & srcFileName,
   const std::string & destFileName
)
{

   const char * arg[3] ;

   arg[0] = "convert" ;
   arg[1] = srcFileName.c_str() ;
   arg[2] = destFileName.c_str() ;

   pid_t procId = fork() ;

   if ( procId == 0 )
   {
       // child process: do exec ....

       execlp("convert",arg[0],arg[1],arg[2],NULL) ;

       // if control returns here, something went wrong:
       // we then inform parent process via the exit status

       exit(errno) ;
   }
   else
   {
       // parent process: wait for child to terminate and then check error

       int status ;
       waitpid(procId,&status,0) ;

       if ( status != 0 )
          throw ParserError("cannot exec 'convert'","ImageLoader::ExecConvert") ;
   }
}

void ImageLoader::CheckAndConvert()
{


   unsigned int pos = fileName.rfind(".tga") ;

   if  ( pos + 4 == fileName.length() )
       return ;

   std::string newFileName = "/tmp/tmp.tga" ;

   //cout << endl << "converting to tga ... " << endl << flush ;

   ExecConvert(fileName,newFileName);

   //cout << "conversion ok." << flush ;

   fileName = newFileName ;

}

} ; // namespace IL




