//*************************************************
//
//  file:     GRFViewer.C
//  function: implementation of class 'GRFViewer'
//
//**********************************************

#include <Modelo3D.H>
#include <GRFViewer.H>
#include <ImageWriter.H>
#include <GRFDiffModel.H>

#include <Fl/Fl_Box.H>
#include <Fl/Fl_Menu_Window.H>
#include <Fl/Fl_Gl_Window.H>
#include <Fl/Fl_Menu_Bar.H>
#include <Fl/gl.h>
#include <Fl/fl_file_chooser.H>
#include <Fl/Fl_Slider.H>
#include <Fl/Fl_Multiline_Output.H>
#include <Fl/fl_message.H>

#include <strstream>

#define DBG false

namespace GRF
{

//***************************************************

class GLDrawingArea :

    public Fl_Gl_Window
{
    public:

    GLDrawingArea
    (
       int         wx,
       int         wy,
       Viewer*     _viewer
    ) ;

    virtual void draw() ;

    int handle( int eventType ) ;

    void OrderSaveImage() ;

    private:

    Viewer*    viewer ;

    double prevx, prevy, wmax ;

    RenderStyleType  prevStyle ;

    void SaveImage() ;


    bool   saveImagePending ;
    std::string imageFileName ;
} ;

//*********************************************

GLDrawingArea::GLDrawingArea
(
   int        wx,
   int        wy,
   Viewer*    _viewer
)
:  Fl_Gl_Window(wx,wy)
,  viewer(_viewer)
,  saveImagePending(false)
{

}

//*************************************************

void GLDrawingArea::OrderSaveImage()
{
   char * fname = fl_file_chooser("File to save:","*.tga",NULL) ;

   if ( fname != NULL )
   {
      imageFileName = fname ;
      saveImagePending = true ;
      redraw() ;
   }
}

//**************************************************

void GLDrawingArea::SaveImage()
{
   int wx = w() ,
       wy = h() ;

   IL::ImageWriter writer(imageFileName,wx,wy);

   GLvoid * pixels = (GLvoid *) (writer.GetPixels()) ;

   glPixelStorei( GL_PACK_ALIGNMENT, 1 ) ;
   glReadPixels(0,0,wx,wy,GL_RGB,GL_UNSIGNED_BYTE,pixels) ;

   writer.SwapRGB() ;
   writer.Write() ;

   saveImagePending = false ;
}

//*************************************************

void GLDrawingArea::draw()
{
   ModeloGRF* model = viewer->GetModel() ;

   if ( model != NULL )
      model->Draw(w(),h());

   if ( saveImagePending )
      SaveImage() ;
}

//**********************************************

int GLDrawingArea::handle
(
   int eventType
)
{
    //std::cout << "GLDrawingArea::handle(" << eventType << ")" << std::cout ;
    //std::cout.flush() ;

    if ( eventType != FL_PUSH && eventType != FL_DRAG && eventType != FL_RELEASE )
       return Fl_Gl_Window::handle(eventType) ;

    Params * params = viewer->GetParams() ;

    if ( eventType == FL_PUSH )
    {
       int iwmax = ( w() > h() ) ? w() : h() ;

       wmax = double(iwmax) ;

       prevx = ( double(Fl::event_x()) / double(w()) ) * params->viewScaleX ;
       prevy = ( double(Fl::event_y()) / double(h()) ) * params->viewScaleY ;

       prevStyle = params->renderStyle ;
    }

    if( eventType == FL_DRAG )
    {
       double
           currx = ( double(Fl::event_x()) / double(w()) ) * params->viewScaleX ,
           curry = ( double(Fl::event_y()) / double(h()) ) * params->viewScaleY ;

       params->shiftX += currx - prevx ;
       params->shiftY += - ( curry - prevy );


       prevx = currx ;
       prevy = curry ;

       params->renderStyle = params->dragRenderStyle ;
    }

    if ( eventType == FL_RELEASE )
    {
       params->renderStyle = prevStyle ;
    }

    redraw() ; // does not redraws now, but says it should be done later

    return 1 ; // tells the GUI we handled the event
}


//**************************************************

void RollerVCallback
(
    Fl_Widget * widget,
    void *      data
)
{
    if (DBG)
    {
      std::cout << "entra RollerVCallback,  widget == " << widget  << " data == " << data << std::cout ;
      std::cout.flush() ;
    }

    Fl_Roller * rollerV = (Fl_Roller *)widget ;
    Viewer *    viewer  = (Viewer *) data ;
    Params *    params  = viewer->GetParams() ;

    params->anguloV = rollerV->value()*180.0*10.0 ;

    viewer->ReDraw() ;
}

//**************************************************

void RollerHCallback
(
    Fl_Widget * widget,
    void *      data
)
{
    if (DBG)
    {
       std::cout << "entra RollerHCallback,  widget == " << widget  << " data == " << data << std::cout ;
       std::cout.flush() ;
    }

    Fl_Roller * rollerH = (Fl_Roller *)widget ;
    Viewer *    viewer  = (Viewer *) data ;
    Params *    params  = viewer->GetParams() ;

    params->anguloH = rollerH->value()*180.0*10.0 ;

    viewer->ReDraw() ;
}

//**************************************************

void GammaSliderCallback
(
    Fl_Widget * widget,
    void *      data
)
{
    if (DBG)
    {
       std::cout << "entra GammaSliderCallback,  widget == " << widget  << " data == " << data << std::cout ;
       std::cout.flush() ;
    }

    Fl_Slider * gammaSlider  = (Fl_Slider *)widget ;
    Viewer *    viewer  = (Viewer *) data ;
    Params *    params  = viewer->GetParams() ;

    params->gamma = gammaSlider->value() ;

    //std::cout << "gamma set to : " << params->gamma << std::cout ;
    std::cout.flush() ;

    viewer->ReDraw() ;
}

//**************************************************

void DesplZSliderCallback
(
    Fl_Widget * widget,
    void *      data
)
{
    if (DBG)
    {
       std::cout << "entra desplZSliderCallback,  widget == " << widget  << " data == " << data << std::cout ;
       std::cout.flush() ;
    }

    Fl_Slider * desplZSlider  = (Fl_Slider *)widget ;
    Viewer *    viewer  = (Viewer *) data ;
    Params *    params  = viewer->GetParams() ;

    params->desplZ = desplZSlider->value() ;

    //std::cout << "despl.z set to : " << params->desplZ << std::cout ;
    std::cout.flush() ;

    viewer->ReDraw() ;
}

//**************************************************

void FocalDisSliderCallback
(
    Fl_Widget * widget,
    void *      data
)
{
    Fl_Slider * focalDisSlider  = (Fl_Slider *)widget ;
    Viewer *    viewer  = (Viewer *) data ;
    Params *    params  = viewer->GetParams() ;

    params->focalDis = focalDisSlider->value() ;

    viewer->ReDraw() ;
}



//***********************************************


void ResetButtonCallback
(
    Fl_Widget * widget,
    void *      data
)
{
    //Fl_Button * resetButton = (Fl_Button *) widget ;
    Viewer *    viewer      = (Viewer *) data ;
    Params *    params      = viewer->GetParams() ;


    params->anguloH = 0.0 ;
    params->anguloV = 0.0 ;

    viewer->GetRollerV()->value( params->anguloV ) ;
    viewer->GetRollerH()->value( params->anguloH ) ;

    viewer->ReDraw() ;
}

//***********************************************


void GammaResetButtonCallback
(
    Fl_Widget * widget,
    void *      data
)
{
    //Fl_Button * resetButton = (Fl_Button *) widget ;
    Viewer *    viewer      = (Viewer *) data ;
    Params *    params      = viewer->GetParams() ;
    Fl_Slider * gammaSlider = viewer->GetGammaSlider() ;


    params->gamma = 1.0 ;

    gammaSlider->value( 1.0 ) ;

    viewer->ReDraw() ;
}


//***********************************************


void OpenCallback
(
    Fl_Widget * widget,
    void *      data
)
{
   Viewer * viewer = (Viewer *) data ;
   viewer->OpenFile() ;
}

//***********************************************


void SaveAsCallback
(
    Fl_Widget * widget,
    void *      data
)
{
   Viewer * viewer = (Viewer *) data ;
   viewer->SaveAs() ;
}


//***********************************************


void QuitCallback
(
    Fl_Widget * widget,
    void *      data
)
{
    exit(0);
}

//***********************************************


void OptsCallback
(
    Fl_Widget * widget,
    void *      data
)
{
    Fl_Menu_Bar * menuBar = (Fl_Menu_Bar *) widget ;
    Viewer *     viewer      = (Viewer *) data ;
    Params *     params      = viewer->GetParams() ;

    int index = menuBar->value() ,
        optNum = -1 ;

    for( int i=0 ; i < 14 ; i++ )
    {
       if ( index == viewer->idx[i] )
       {
          optNum = i ;
          break ;
       }
    }

    if ( optNum == -1 )
       return ;

    switch ( optNum )
    {
       case 0 :
          params->reverseFacesOrientation = ! params->reverseFacesOrientation ;
          break ;

       case 1:
          params->reverseNormals = ! params->reverseNormals ;
          break ;

       case 2:
          params->drawNormals = ! params->drawNormals ;
          break ;

       case 3:
          params->drawBox = ! params->drawBox ;
          break ;

       case 4:
          params->renderStyle = rsPoints ;
          break ;

       case 5:
          params->renderStyle = rsWireframe ;
          break ;

       case 6:
          params->renderStyle = rsSolidFlatShaded ;
          break ;

       case 7:
          params->renderStyle = rsSolidSmoothShaded ;
          break ;

       case 8:
          params->renderStyle = rsIradiance ;
          break ;

       case 9:
          params->dragRenderStyle = rsPoints ;
          break ;

       case 10:
          params->dragRenderStyle = rsWireframe ;
          break ;

       case 11:
          params->dragRenderStyle = rsSolidFlatShaded ;
          break ;

       case 12:
          params->dragRenderStyle = rsSolidSmoothShaded ;
          break ;

       case 13:
          params->dragRenderStyle = rsIradiance ;
          break ;
    }


    viewer->ReDraw() ;
}

//*************************************************

void InfoCallback
(
    Fl_Widget * widget,
    void *      data
)
{
    Viewer * viewer  = (Viewer *) data ;

    viewer->DisplayInfoWindow() ;
}

//*************************************************

void AboutCallback
(
    Fl_Widget * widget,
    void *      data
)
{
    Viewer * viewer  = (Viewer *) data ;

    viewer->DisplayAboutWindow() ;
}

//*************************************************

void SaveImageCallback
(
    Fl_Widget * widget,
    void *      data
)
{
    Viewer * viewer  = (Viewer *) data ;

    viewer->SaveImage() ;
}

//*******************************************************

void LoadCameraCallback
(
    Fl_Widget * widget,
    void *      data
)
{
    Viewer * viewer  = (Viewer *) data ;

    viewer->LoadCamera() ;
}

//************************************************************

void SaveCameraCallback
(
    Fl_Widget * widget,
    void *      data
)
{
    Viewer * viewer  = (Viewer *) data ;

    viewer->SaveCamera() ;
}

//************************************************************

void CompareCallback
(
    Fl_Widget * widget,
    void *      data
)
{
    Viewer * viewer  = (Viewer *) data ;

    viewer->Compare() ;
}


//************************************************************

void RunGirtCallback
(
    Fl_Widget * widget,
    void *      data
)
{
    Viewer * viewer  = (Viewer *) data ;

    viewer->RunGirt() ;
}

//************************************************************

void EditGirtOptsCallback
(
    Fl_Widget * widget,
    void *      data
)
{
    Viewer * viewer  = (Viewer *) data ;

    viewer->EditGirtOpts() ;
}

//************************************************************

void LoadGirtOptsCallback
(
    Fl_Widget * widget,
    void *      data
)
{
    Viewer * viewer  = (Viewer *) data ;

    viewer->LoadGirtOpts() ;
}

//************************************************************

void SaveGirtOptsCallback
(
    Fl_Widget * widget,
    void *      data
)
{
    Viewer * viewer  = (Viewer *) data ;

    viewer->SaveGirtOpts() ;
}


//**************************************************************
//
//   class VIEWER
//
//**************************************************************


void Viewer::Compare()
{

   char *    compFileName ;
   Parser *  parser2 ;

   try
   {
      if (! model->GetParser()->AnyVertexWithIradiance() )
         throw ParserError("current model has no irradiance information","trying to compare current model");

      compFileName = fl_file_chooser("Select a grf file to compare","{*.grf|*.igrf}",NULL);

      if ( compFileName == NULL )
         return ;

      std::cout << std::flush
           << std::cout << "*******************************"
           << std::cout << "*** Comparing current (" << fileName << ") with " << compFileName
           << std::cout
           << std::flush ;

      parser2 = new Parser(compFileName);

      if ( parser2 == NULL )
         throw ParserError("'new' failed cannot create 'parser2'","GRFViewer::Compare") ;

      CompareWith( parser2 ) ;
   }
   catch( ParserError err )
   {
      DisplayErrorWindow(err);
   }

   if ( parser2 != NULL )
      delete parser2 ;

   std::cout << std::cout << "*** End comparision"
        << std::cout << "*******************"
        << std::cout << std::flush ;

}

//********************************************************

void Viewer::CompareWith
(
   const Parser * parser2
)
{
   const Parser * parser1 = model->GetParser() ;

   DiffModel dm(parser1,parser2) ;

   dm.WriteDifferences() ;
}


//********************************************************

void Viewer::CloseAboutWindow()
{

  if ( aboutWindow != NULL )
  {
     delete aboutWindow ;
     aboutWindow = NULL ;
  }
}

//*****************************************************

void Viewer::CloseGirtOptsWindow()
{
  if ( girtOptsWindow != NULL )
  {
     delete girtOptsWindow ;
     girtOptsWindow = NULL ;
  }
}

//*******************************************************

void Viewer::CloseInfoWindow()
{
  if ( infoWindow != NULL )
  {
     delete infoWindow ;
     infoWindow = NULL ;
  }
}


//**************************************************************

const int defFont = FL_HELVETICA_BOLD_ITALIC ;
const int defColor = fl_color_cube(0,0,2);

//**************************************************************

void Viewer::CreateWidgets()
{
    const int wx = 512 ;
    const int wy = 512 ;
    const int rollW = 25 ;
    const int rollH = 120  ;



    const int menuBarH = 25 ;
    const int glAreaBorder = 2 ;

    const int fwx = wx+glAreaBorder*2 ;
    const int fwy = wy+glAreaBorder*2 ;

    const int yorg = menuBarH+fwy ;



    //
    // create main window
    //

    mainWindow = new Fl_Window(fwx+rollW+10,
                               fwy+rollW+10+menuBarH,
                               fileName);
    //
    // create the frame around the gl region
    //

    Fl_Group * glAreaFrame = new Fl_Group( 0,menuBarH,fwx,fwy ) ;

    //
    // create the gl region
    //

    glArea = new GLDrawingArea(wx,wy,this);

    // check

    if ( glArea == NULL || mainWindow == NULL || glAreaFrame == NULL )
       throw ParserError("'new' failed, cannot create window","Viewer::CreateWidgets");


    //
    // create widgets inside main window
    //

    mainWindow->begin() ;


       rollerV        = new Fl_Roller(fwx+5,yorg-rollH,rollW,rollH);
       rollerH        = new Fl_Roller(fwx-rollH,yorg+5,rollH,rollW);
       gammaSlider    = new Fl_Slider( 10,yorg+5,rollH,rollW) ;
       desplZSlider   = new Fl_Slider( fwx+5,menuBarH+5,rollW,rollH);
       focalDisSlider = new Fl_Slider( fwx+5,menuBarH+5+rollH+10,rollW,rollH);

       Fl_Button *   resetButton      = new Fl_Button( fwx+5,yorg+5,rollW, rollW, "R" ) ;
       Fl_Button *   gammaResetButton = new Fl_Button( 10+rollH+10, yorg+5, rollW, rollW, "R" );
       Fl_Menu_Bar * menuBar          = new Fl_Menu_Bar(0,0,fwx,menuBarH);

       if (  rollerV == NULL
          || rollerH == NULL
          || resetButton == NULL
          || glAreaFrame == NULL
          || menuBar == NULL
          || gammaSlider == NULL
          || gammaResetButton == NULL
          || desplZSlider == NULL
          || focalDisSlider == NULL
          )
          throw ParserError("'new' failed, cannot create widget","Viewer::CreateWidgets");

    mainWindow->end() ;

    //
    //  set attributes for widgets and windows
    //

    rollerV->type(FL_VERTICAL);
    rollerV->range(-1.0,1.0);
    rollerV->value(0.0);

    rollerV->callback(RollerVCallback);
    rollerV->user_data( (void *)this ) ;

    rollerH->type(FL_HORIZONTAL);
    rollerH->range(-1.0,1.0);
    rollerH->value(0.0);

    rollerH->callback(RollerHCallback);
    rollerH->user_data( (void *)this ) ;

    resetButton->callback( ResetButtonCallback ) ;
    resetButton->user_data( (void *) this ) ;
    resetButton->labelfont( defFont ) ;
    resetButton->labelcolor( defColor );
    resetButton->box( FL_ROUND_UP_BOX ) ;

    glArea->position(glAreaBorder,menuBarH+glAreaBorder);

    glAreaFrame->box( FL_DOWN_BOX ) ;

    gammaSlider->callback( GammaSliderCallback ) ;
    gammaSlider->user_data( (void *) this ) ;
    gammaSlider->type( FL_HOR_NICE_SLIDER ) ;
    gammaSlider->range(0.0001,5.0);
    gammaSlider->value(1.0);


    gammaResetButton->callback( GammaResetButtonCallback ) ;
    gammaResetButton->user_data( (void *) this ) ;
    gammaResetButton->labelfont( defFont ) ;
    gammaResetButton->labelcolor( defColor );
    gammaResetButton->box( FL_ROUND_UP_BOX ) ;
    //gammaSlider->box( FL_NO_BOX ) ;

    desplZSlider->callback( DesplZSliderCallback ) ;
    desplZSlider->user_data( (void *) this ) ;
    desplZSlider->type( FL_VERT_NICE_SLIDER ) ;
    desplZSlider->range(4.0,0.0);
    desplZSlider->value(0.0);
    //focusLengthSlider->box( FL_NO_BOX ) ;

    focalDisSlider->callback( FocalDisSliderCallback ) ;
    focalDisSlider->user_data( (void *) this ) ;
    focalDisSlider->type( FL_VERT_NICE_SLIDER ) ;
    focalDisSlider->range(4.0,0.2);
    focalDisSlider->value(1.0);
    //focusLengthSlider->box( FL_NO_BOX ) ;

    void * tv = (void *) this ;

    menuBar->textfont( defFont ) ;
    menuBar->labelcolor( defColor ) ;
    menuBar->textcolor( defColor ) ;

    menuBar->add("File/Open file" ,  0, OpenCallback,       tv, 0);
    menuBar->add("File/Save as",     0, SaveAsCallback,     tv, 0);
    menuBar->add("File/Load camera", 0, LoadCameraCallback, tv, 0);
    menuBar->add("File/Model info",  0, InfoCallback,       tv, 0);
    menuBar->add("File/Save image",  0, SaveImageCallback,  tv, 0);
    menuBar->add("File/Save camera", 0, SaveCameraCallback, tv, 0);
    menuBar->add("File/Compare",     0, CompareCallback,    tv, 0);
    menuBar->add("File/Quit" ,       0, QuitCallback,       tv, 0);

    menuBar->add("Girt/Run",          0, RunGirtCallback,      tv,0);
    menuBar->add("Girt/Edit options", 0, EditGirtOptsCallback, tv,0);
    menuBar->add("Girt/Load options", 0, LoadGirtOptsCallback, tv,0);
    menuBar->add("Girt/Save options", 0, SaveGirtOptsCallback, tv,0);



    idx[0] = menuBar->add("Options/Reverse faces orientation", 0,OptsCallback,tv, FL_MENU_TOGGLE ) ;
    idx[1] = menuBar->add("Options/Reverse normals",           0,OptsCallback,tv, FL_MENU_TOGGLE ) ;
    idx[2] = menuBar->add("Options/Draw normals",              0,OptsCallback,tv, FL_MENU_TOGGLE ) ;
    idx[3] = menuBar->add("Options/Draw bounding box",         0,OptsCallback,tv, FL_MENU_TOGGLE ) ;

    idx[4] = menuBar->add("Style/Points",        0,OptsCallback,tv, FL_MENU_RADIO ) ;
    idx[5] = menuBar->add("Style/Wireframe",     0,OptsCallback,tv, FL_MENU_RADIO ) ;
    idx[6] = menuBar->add("Style/Flat shaded",   0,OptsCallback,tv, FL_MENU_RADIO ) ;
    idx[7] = menuBar->add("Style/Smooth shaded", 0,OptsCallback,tv, FL_MENU_RADIO ) ;
    idx[8] = menuBar->add("Style/Irradiance",    0,OptsCallback,tv, FL_MENU_RADIO ) ;


    idx[9]  = menuBar->add("Style/Drag style/Points",    0,     OptsCallback, tv, FL_MENU_RADIO );
    idx[10] = menuBar->add("Style/Drag style/Wireframe", 0,     OptsCallback, tv, FL_MENU_RADIO );
    idx[11] = menuBar->add("Style/Drag style/Flat Shaded", 0,   OptsCallback, tv, FL_MENU_RADIO );
    idx[12] = menuBar->add("Style/Drag style/Smooth shaded", 0, OptsCallback, tv, FL_MENU_RADIO );
    idx[13] = menuBar->add("Style/Drag style/Irradiance", 0,    OptsCallback, tv, FL_MENU_RADIO );

    menuBar->add("About",0,AboutCallback, tv, 0);




    //
    // add sub-windows to main window
    //

    glAreaFrame->add( glArea ) ;
    glAreaFrame->resizable( glArea ) ;

    mainWindow->add( glAreaFrame ) ;
    mainWindow->resizable( glAreaFrame ) ;

    //
    // show all
    //

    mainWindow->show();
    glArea->show() ;


}

//***********************************************

void Viewer::DestroyWidgets()
{

}

//*********************************************


void Viewer::DisplayAboutWindow()
{

   if ( aboutWindow == NULL )
   {
      aboutWindow = new AboutWindow(this);

      if ( aboutWindow == NULL )
         throw ParserError("cannot create about window","Viewer::DisplayAboutWindow");
   }

}


//*********************************************


void Viewer::DisplayInfoWindow()
{

   if ( infoWindow == NULL )
   {
      infoWindow = new InfoWindow(this);

      if ( infoWindow == NULL )
         throw ParserError("cannot create info window","Viewer::DisplayInfoWindow");
   }

}


//***********************************************

void Viewer::DisplayErrorWindow
(
   const ParserError & err
)
{

   std::string s = "   Error: " ;

   s += err.GetMessage() ;
   s += "\n" ;
   s += "   (" ;
   s += err.GetWhere() ;
   s += ")" ;

   fl_message(s.c_str());
}

//***********************************************

void Viewer::EditGirtOpts()
{
   if ( girtOptsWindow == NULL )
   {
      girtOptsWindow = new GirtOptsWindow(this);

      if ( girtOptsWindow == NULL )
         throw ParserError("cannot create Girt options window","Viewer::EditGirtOpts");
   }
}

//***********************************************

void Viewer::LoadCamera()
{
   char * fname = fl_file_chooser("Select a file to load","*.cam",NULL);

   if ( fname == NULL )
      return ;


   std::ifstream is(fname) ;

   if ( ! is.good() )
   {
      std::string s = "cannot open file: " ;
      s += fname ;
      DisplayErrorWindow(ParserError(s,"Viewer::LoadCamera"));
      return ;
   }

   Params * p = model->GetParams() ;
   int wx,wy ;

   is  >> p->anguloV
       >> p->anguloH
       >> p->shiftX
       >> p->shiftY
       >> p->desplZ
       >> p->focalDis
       >> wx
       >> wy ;



   is.close() ;

   mainWindow->size(wx,wy) ;

   ReDraw() ;
}

//**********************************************

void Viewer::LoadGirtOpts()
{

}

//***********************************************

void Viewer::LoadModel
(
   const char * _fileName
)
{
   if ( model != NULL )
      delete model ;

   fileName = _fileName ;

   model = new ModeloGRF(fileName);

   if ( model == NULL )
      throw ParserError("'new' failed cannot create 'model'","GRFViewer::GRFViewer") ;
}

//***********************************************************

void Viewer::OpenFile()
{
   char * fname = fl_file_chooser("Select a file","*.{grf|igrf}",NULL);

   ModeloGRF * oldModel ;

   if ( fname != NULL )
   {
      try
      {
         oldModel = model ;
         model = NULL ;

         CloseInfoWindow() ;
         LoadModel( fname ) ;
         ResetWidgets() ;
         ReDraw();

         delete oldModel ;
      }
      catch( ParserError err )
      {
         if ( model != NULL )
            delete model ;

         model = oldModel ;

         DisplayErrorWindow(err);
      }
   }
}

//*******************************************************

void Viewer::PrintInfo
(
   std::ostream & os
)
{
  if ( model != NULL )
  {
     os << std::cout << "file : " << fileName
        << std::cout << "----------------------------------------" ;

     model->GetModelInfo(os);
  }
  else
     os << "no file loaded" << std::cout ;

}

//*********************************************************

void Viewer::ReDraw()
{
   glArea->redraw() ;
}

//*********************************************************

void Viewer::ResetWidgets()
{
   if ( rollerV != NULL )
      rollerV->value(0.0) ;

   if ( rollerH != NULL )
      rollerH->value(0.0);

   if ( desplZSlider != NULL )
      desplZSlider->value(0.0) ;

   if ( gammaSlider != NULL )
      gammaSlider->value(1.0) ;
}

//**********************************************

void Viewer::RunGirt()
{

}

//*******************************************************

void Viewer::SaveAs()
{
   char * fname = fl_file_chooser("Save GRF model as:","*.cam",NULL);

   if ( fname == NULL )
      return ;

   model->SaveAs( fname ) ;
}


//*******************************************************

void Viewer::SaveCamera()
{
   char * fname = fl_file_chooser("Select a file to save","*.cam",NULL);

   if ( fname == NULL )
      return ;


   std::ofstream os(fname) ;

   if ( ! os.good() )
   {
      std::string s = "cannot write onto file: " ;
      s += fname ;
      throw ParserError(s,"Viewer::SaveCamera");
   }

   Params * p = model->GetParams() ;

   os          << p->anguloV
       << std::cout << p->anguloH
       << std::cout << p->shiftX
       << std::cout << p->shiftY
       << std::cout << p->desplZ
       << std::cout << p->focalDis
       << std::cout << mainWindow->w()
       << std::cout << mainWindow->h()
       ;

   os.flush() ;

   os.close() ;


}

//**********************************************

void Viewer::SaveGirtOpts()
{

}


//*********************************************************

void Viewer::SaveImage()
{
   glArea->OrderSaveImage();
}

//*********************************************************

Viewer::Viewer
(
   const char * _fileName,
   bool _verbose
)
:  fileName(NULL)
,  verbose(_verbose)
,  model(NULL)
,  rollerV(NULL)
,  rollerH(NULL)
,  desplZSlider(NULL)
,  gammaSlider(NULL)
,  glArea(NULL)
,  mainWindow(NULL)
,  infoWindow(NULL)
,  girtOptsWindow(NULL)
{
   LoadModel(_fileName) ;
   CreateWidgets() ;
}

//*******************************************************

Viewer::~Viewer()
{

   DestroyWidgets() ;

   if ( model != NULL )
      delete model ;
}






} ; // namespace GRF

