//***************************************************************
//***
//***   File:     GRFTexture.C
//***   Function: Implementation of class 'Texture'
//***
//***   2-Nov-01      Carlos      Creation.
//***   
/* 2002-06-17:
 * Replaced all references to variables defined in namespace std, which
 * were used with no prefix, with std::<reference>
 * in order to compile with gcc 3.1.1, which no longer has 
 * "using namespace standard"
 * Rubn J. Garca Hernndez
 */ 
//***************************************************************


#include <math.h>
#include <GL/gl.h>

#include <ImageLoader.H>
#include <ParserError.H>
#include <GRFTexture.H>

namespace GRF
{

using namespace IL ;

//*********************************************************

Texture::Texture
(
   const std::string & _fileName
)
{

   fileName = _fileName ;

   ImageLoader * il = new ImageLoader(fileName);

   if ( il == NULL )
      throw ParserError("'new' failed","Texture::Texture");

   nxImg = il->XSize() ;
   nyImg = il->YSize() ;

   nxTex = 1 ;
   nyTex = 1 ;

   while( nxTex < nxImg )
       nxTex *= 2 ;

   while( nyTex < nyImg )
       nyTex *= 2 ;

   factorx = (double) nxImg / (double) nxTex ;
   factory = (double) nyImg / (double) nyTex ;

   size_t  pixelsSize = (size_t)3 * (size_t)nxTex * (size_t)nyTex ;

   pixels = new (unsigned char) [pixelsSize] ;

   if ( pixels == NULL )
      throw ParserError("'new' failed, cannot alloc texture pixels","GRF::Texture::Texture");

   for( unsigned long ix = 0 ; ix < nxImg ; ix++ )
   for( unsigned long iy = 0 ; iy < nyImg ; iy++ )
   {
       unsigned char pixel[3] ;

       il->GetPixel(ix,iy,pixel);

       size_t offset = size_t(3)*( size_t(nxTex*iy) + size_t(ix)) ;

       pixels[offset++] = pixel[0] ;
       pixels[offset++] = pixel[1] ;
       pixels[offset++] = pixel[2] ;
   }

   delete il ;
}

//******************************************************************

void Texture::GetRGB
(
   double u,
   double v,
   double rgb[3]
)
   const
{
   double
      dix = floor( u*double(nxImg) ),
      diy = floor( v*double(nyImg) ) ;

   unsigned long
      ix = (unsigned long) dix ,
      iy = (unsigned long) diy ;

   ix = ( ix < nxImg ) ? ix : nxImg-1 ;
   iy = ( iy < nyImg ) ? iy : nyImg-1 ;

   size_t offset = size_t(3)*( size_t(nxTex*iy) + size_t(ix)) ;

   const double imax = double(1.0)/double(255) ;

   rgb[0] =  imax * double( pixels[offset++] ) ;
   rgb[1] =  imax * double( pixels[offset++] ) ;
   rgb[2] =  imax * double( pixels[offset++] ) ;
}

//********************************************************************

void Texture::GL_ActivateTexture
()
   const
{

   glPixelStorei(GL_UNPACK_ALIGNMENT,1 ) ;

   glTexImage2D( GL_TEXTURE_2D,      // target
                 GLint(0),           // level
                 GL_RGB,             // internal format
                 GLsizei(nxTex),     // width
                 GLsizei(nyTex),     // height
                 GLint(0),           // border
                 GL_RGB,             // format
                 GL_UNSIGNED_BYTE,   // type
                 (GLvoid *) pixels   // pixels
               ) ;

   glTexParameterf( GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP ) ;
   glTexParameterf( GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP ) ;

   glTexParameterf( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR ) ;
   glTexParameterf( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR ) ;
   glTexEnvi(GL_TEXTURE_ENV,GL_TEXTURE_ENV_MODE,GL_MODULATE);
   glEnable(GL_TEXTURE_2D) ;
}

void Texture::GL_SetTextureCoordinates
(
   double u ,
   double v
)
   const
{
   glTexCoord2d( u*factorx, v*factory ) ;
}

const std::string & Texture::GetFileName
()
   const
{
   return fileName ;
}

} ; // namespace GRF


