#ifndef GRF_PARSER_H
#define GRF_PARSER_H


#include <vector>
#include <Vector>
#include <Token.H>
#include <ParserError.H>
#include <Face.H>
#include <GRFVertex.H>
#include <Light.H>
#include <fstream>
#include <GRFTexture.H>



namespace GRF
{

using namespace Vector ;

class TrStackNode ;
class DefStackNode ;
class InputStack ;

//*****************************************************

class Parser
{
   public :

   Parser
   (
      const char * _fileName,
      bool         _verbose = true
   )  ;

   ~Parser() ;

   const Face * GetFace
   (
      unsigned long iFace
   )
      const ;

   const Vertex * GetVertex
   (
      unsigned long iVertex
   )
      const ;

   const Light * GetLight
   (
      unsigned long iLight
   )
      const ;


   const Texture * GetTexture
   (
      unsigned long iTexture
   )
      const ;

   unsigned long NVertexs
   ()
      const ;

   bool AnyVertexWithIradiance
   ()
      const ;

   unsigned long NFaces
   ()
      const ;

   unsigned long NLights
   ()
      const ;

   unsigned long NTextures
   ()
      const ;

   bool AnyVertexWithNormal
   ()
      const ;

   void ComputeVertexNormals
   () ;

   void SaveAs
   (
      const char * fileName
   )
      const ;

   //***************************************************************

   private:

   Token           tk ; // look-ahead token

   const char *    fileName ;

   bool            verbose  ;

   InputStack *    stack    ; // stack of input streams (for multiple-level included files)
   DefStackNode *  defStack ; // stack of entities definitions (textures, colors, etc..)
   TrStackNode *   trStack  ; // stack of transformation matrixes

   double *        pCurrentFaceColorRGB ;
      // pointer to rgb-std::vector with current face color, NULL if none

   double          currentFaceColorRGB[3] ;
      // current face color

   std::vector<Vertex>    vertexs ;
   std::vector<Face>      faces   ;
   std::vector<Light>     lights  ;
   std::vector<Texture>   textures ;

   unsigned long nLights ;
      // total number of extended light sources (already processed and stored)

   unsigned long nVertexs ;
     // total number of vertex (already processed and stored)

   unsigned long nFaces   ;
     // total number of faces (already processed and stored)

   unsigned long nMeshVertexs ;
     // number of vertex in current mesh (already processed and stored)

   unsigned long meshVertexIndexBase ;
     // base or shift to add when converting from
     // mesh-relative vertex indexes to absolute vertex indexes
     // (that is: absolute vertex index of the vertex whose relative index
     //  in the current mesh is 0).

   unsigned long nTextures ;
     // number of textures actually defined

   long int currTextNum ;
     // number of texture currently in use, -1 if no texture in use
     // is to be interpreted as an index for std::vector 'textures'
     // initialized for each mesh to -1


   //***************************************************


   void OpenStream( const std::string & fileName ) ;

   void CloseStream() ;

   void ReadBlock() ;

   void ReadMesh() ;

   void ReadMeshVertexs() ;

   void ReadMeshFaces() ;

   void ReadLight() ;

   void ReadVertex() ;

   void ReadFace() ;

   void ReadFace2() ;

   void ReadFacesColor() ;

   void ReadInclude() ;

   void ReadTransformBlock() ;

   void ReadDefinitionBlock() ;

   void ReadTextureDefinition
   (
      const std::string & textureId
   ) ;

   void LoadTexture
   (
      const std::string & textureId ,
      const std::string & fileName
   ) ;

   void LoadNextToken() ;

   void ReadSingleToken
   (
      TokenType type
   ) ;

   void ReadSingleNumericToken() ;

   void ReadSingleTokenOpt
   (
      TokenType type
   ) ;

   void ReadVector3
   (
      double v[3]
   ) ;

   void ReadVector2
   (
      double v[2]
   ) ;

   void PushTransform
   (
      const Transformd & tr
   ) ;

   void PopTransform() ;

   void PushDefinitionsStackMark();

   void PopDefinitionsStack();  // pop until a mark is found, or error

   void DeleteDefStackNode
   (
       DefStackNode * n
   ) ;

   void ReadTextureUse() ;

} ; // class Parser

//**********************************************************

class InputStack
{
   public:

   std::ifstream *          is ;
   struct InputStack * next ;
}   ;

//*******************************************************

class TrStackNode   // transformations stack node
{
   public:

   Transformd    transform ;
   TrStackNode * next ;
} ;

//******************************************************

typedef enum { dfMark, dfTexture } DefinitionType ;

class DefStackNode // definitions stack node
{
   public:

   DefinitionType type ;
   std::string         identifier ;
   void *         data ;
   DefStackNode * next ;

   DefStackNode
   (
       DefinitionType itype ,
       const std::string & iidentifier ,
       void *         idata ,
       DefStackNode * inext
   )
   :   type(itype)
   ,   identifier(iidentifier)
   ,   data(idata)
   ,   next(inext)
   {}

} ;

   
   
} ; // namespace GRF
   
#endif // GRF_PARSER_H
