#include <GRFParser.H>
#include <string>
#include <stdio.h>

#include <Vector>
#include <ImageLoader.H>

using namespace Vector ;
using namespace IL ;

namespace GRF
{

Parser::Parser
(
   const char * _fileName,
   bool         _verbose
)
:  fileName(_fileName)
,  verbose(_verbose)

,  stack(NULL)
,  defStack(NULL)
,  trStack(NULL)

,  nLights(0L)
,  nVertexs(0L)
,  nFaces(0L)
,  nTextures(0L)

,  currTextNum(0L)

{
   OpenStream(fileName);

   // get look-ahead token for the first time

   LoadNextToken() ;

   // a grf file is a (maybe empty) list of blocks:

   while ( tk.Type() != tkEndOfInput )
      ReadBlock() ;


   // compute normals if not present

   if ( ! AnyVertexWithNormal() )
      ComputeVertexNormals() ;
}

//*************************************************************


void Parser::OpenStream
(
   const std::string & fileName
)
{
   //cout << endl << "opening include file (" << fileName << ") ..." << flush ;

   InputStack * res ;

   res = new InputStack ;

   if ( res == NULL )
        throw ParserError("new failed","Parser::OpenStream");

   res->is = new std::ifstream(fileName.c_str()) ;

   if ( res->is == NULL )
      throw ParserError("cannot alloc or initialize","creating parser") ;

   if ( ! res->is->good() )
   {
      std::string s = "cannot open file (" + fileName + ") ";
      throw ParserError(s,"Parser::OpenStream") ;
   }


   res->next = stack ;
   stack = res ;
}

//*************************************************************


void Parser::CloseStream()
{
   InputStack * tmp = stack ;

   stack = stack->next ;

   delete tmp ;
}

//*************************************************************

void Parser::LoadNextToken()
{
  if ( stack == NULL )
     throw ParserError("internal error: empty stack","Parser::LoadNextToken");

  *(stack->is) >> tk ;

  while ( tk.Type() == tkEndOfInput && stack->next != NULL )
  {
      CloseStream() ;
      *(stack->is) >> tk ;
  }
}

//************************************************************

void Parser::ReadBlock()
{
   // a block is either a mesh, a light, an include sentence,
   // a transform block, or a definition block

   if ( tk.Type() == tkBegin )
       ReadMesh() ;
   else if ( tk.Type() == tkLight )
       ReadLight() ;
   else if ( tk.Type() == tkInclude )
       ReadInclude() ;
   else if ( tk.Type() == tkTransform )
       ReadTransformBlock() ;
   else if ( tk.Type() == tkDef )
       ReadDefinitionBlock() ;
   else
       throw ParserError("expected one of: begin,light,include,transform,def","Parser::ReadBlock") ;
}

//************************************************************

void Parser::ReadTransformBlock()
{
   Transformd t ; // == identity (default constructor)


   //cout << "comienza transform, t ==" << t.GetMDir() ;

   ReadSingleToken(tkTransform) ;

   double v[3] ;

   while ( tk.Type() == tkScale ||
           tk.Type() == tkRotation ||
           tk.Type() == tkTranslation
         )
   {
         switch( tk.Type() )
         {
         case tkScale :

             ReadSingleToken( tkScale ) ;
             ReadVector3( v ) ;

             t *= TScale(Vector3d(v[0],v[1],v[2])) ;

             break ;

         case tkRotation :

             ReadSingleToken( tkRotation ) ;
             ReadVector3( v ) ;

             t *= TRotation(X,v[0]) ;
             t *= TRotation(Y,v[1]) ;
             t *= TRotation(Z,v[2]) ;

             break ;

         case tkTranslation :

             ReadSingleToken( tkTranslation ) ;
             ReadVector3( v ) ;

             t *= TTranslation(Vector3d(v[0],v[1],v[2])) ;

             break ;

         default :
             throw ParserError("it is imposible to reach this point","GRFParser::ReadTransformBlock");
             break ;
         }
   }

   ReadSingleToken(tkBegin) ;

   PushTransform( t ) ;


   //cout << "ejecutado 'push transform', trStack->transform == " << trStack->transform.GetMDir() ;

   while( tk.Type() != tkEnd )
      ReadBlock() ;

   PopTransform() ;

   ReadSingleToken(tkEnd) ;
}

//***********************************************************

void Parser::ReadDefinitionBlock()
{
   ReadSingleToken(tkDef);

   PushDefinitionsStackMark() ;

   while( tk.Type() == tkIdentifier )
   {

       std::string identifier = tk.ValueString() ;

       ReadSingleToken(tkIdentifier);

       if ( tk.Type() != tkEqual )
          throw ParserError("'=' expected after identifier","Parser::ReadDefinitionBlock");

       ReadSingleToken(tkEqual);

       switch( tk.Type() )
       {
          case tkTexture :

             ReadTextureDefinition(identifier) ;
             break ;

          default:

             throw ParserError("definition type (texture/...) expected after '='","Parser::ReadDefinitionBlock");
       }
   }


   if ( tk.Type() != tkBegin )
        throw ParserError("'begin' expected after definitions section","Parser::ReadDefinitionBlock") ;

   ReadSingleToken( tkBegin ) ;

   while( tk.Type() != tkEnd )
      ReadBlock() ;

   PopDefinitionsStack() ;

   ReadSingleToken( tkEnd ) ;
}

//************************************************************

void Parser::ReadTextureDefinition
(
   const std::string & textureId
)
{
   ReadSingleToken(tkTexture);

   if ( tk.Type() != tkLiteral )
      throw ParserError("filename expected after 'texture'","Parser::ReadTextureDefinitionBlock");

   std::string fileName = tk.ValueString() ;

   LoadTexture(textureId,fileName);

   ReadSingleToken(tkLiteral);
}


//*************************************************************

void Parser::LoadTexture
(
   const std::string & textureId ,
   const std::string & fileName
)
{
   // get texture number for new texture

   unsigned long textureNumber = nTextures ;

   unsigned long * data = new unsigned long (textureNumber) ;

   if ( data == NULL )
      throw ParserError("'new' failed -- cannot alloc 8 bytes !!","Parser::LoadTexture");

   nTextures++ ;


   // load texture

   Texture tex(fileName) ;

   // push definition stack node

   DefStackNode * n = new DefStackNode(dfTexture,textureId,(void *)data,defStack) ;

   if ( n == NULL )
      throw ParserError("'new' failed, cannot create 'DefStackNode'","Parser::LoadTexutre");

   defStack = n ;


   // append texture to 'textures' vector

   textures.push_back( tex ) ;

}

//************************************************************

void Parser::ReadInclude()
{
   ReadSingleToken(tkInclude);

   if ( tk.Type() != tkLiteral )
      throw ParserError("literal expected after 'include' (did you used '\"' ?)","Parser::ReadInclude");

   std::string fileName = tk.ValueString() ;

   OpenStream(fileName);
   ReadSingleToken(tkLiteral);

}

//************************************************************

void Parser::ReadMesh()
{

   pCurrentFaceColorRGB = NULL ;

   meshVertexIndexBase = nVertexs ;
   nMeshVertexs = 0L ;

   ReadSingleToken( tkBegin ) ;
   ReadSingleTokenOpt( tkMesh ) ;

   currTextNum = -1 ;

   if ( tk.Type() == tkTexture )
      ReadTextureUse() ;

   ReadMeshVertexs() ;
   ReadMeshFaces() ;

   ReadSingleToken( tkEnd ) ;
   ReadSingleTokenOpt( tkMesh ) ;
}

//************************************************************

void Parser::ReadTextureUse()
{
   ReadSingleToken(tkTexture) ;

   if ( tk.Type() != tkIdentifier )
      throw ParserError("identifier expected after 'texture'","Parser::ReadTextureUse");


   std::string textureIdent = tk.ValueString() ;


   long int textureNumber   = -1 ;


   if ( textureIdent != "none" )
   {
      DefStackNode * pDefStack = defStack ;

      while( pDefStack != NULL )
      {
         if ( pDefStack->type == dfTexture )
         if ( pDefStack->identifier == textureIdent )
         {
            textureNumber = (long int) (* ((unsigned long *) pDefStack->data )) ;
            break ;
         }

         pDefStack = pDefStack->next ;
      }

      if ( textureNumber == -1 )
      {
         std::string s = "" ;
         s += "texture '" + textureIdent  + "' not found" ;
         throw ParserError (s,"Parser::ReadTextureUse");
      }
   }

   ReadSingleToken(tkIdentifier);

   currTextNum = textureNumber ;

}

//************************************************************

void Parser::ReadLight()
{
   ReadSingleToken(tkLight);
   ReadSingleToken(tkBegin) ;

   double org[3], edge0[3], edge1[3], rgb[3] ;

   ReadVector3(org);
   ReadVector3(edge0);
   ReadVector3(edge1);

   if ( trStack != NULL )
   {
      Vector3d p(org[0],org[1],org[2]) ,
               p0(org[0]+edge0[0],org[1]+edge0[1], org[2]+edge0[2]) ,
               p1(org[0]+edge1[0],org[1]+edge1[1], org[2]+edge1[2]) ;

      p  *= trStack->transform ;
      p0 *= trStack->transform ;
      p1 *= trStack->transform ;

      org[0] = p(X) ;
      org[1] = p(Y) ;
      org[2] = p(Z) ;

      edge0[0] = p0(X)-p(X) ;
      edge0[1] = p0(Y)-p(Y) ;
      edge0[2] = p0(Z)-p(Z) ;

      edge1[0] = p1(X)-p(X) ;
      edge1[1] = p1(Y)-p(Y) ;
      edge1[2] = p1(Z)-p(Z) ;
   }

   rgb[0] = 1.0 ;
   rgb[1] = 1.0 ;
   rgb[2] = 1.0 ;

   if ( tk.Type() == tkRGB )
   {
      ReadSingleToken(tkRGB) ;
      ReadVector3(rgb) ;
   }

   ReadSingleToken(tkEnd);
   //ReadSingleTokenOpt(tkLight);

   Light l(org,edge0,edge1,rgb) ;

   lights.push_back(l) ;

   nLights++ ;

}

//************************************************************

void Parser::ReadSingleToken
(
   TokenType type
)
{
   if ( tk.Type() != type )
      throw ParserError("expected token not found","Parser::ReadSingleToken") ;

   LoadNextToken() ;
}

//************************************************************

void Parser::ReadSingleNumericToken()
{
   if ( tk.Type() != tkNumericInteger &&
        tk.Type() != tkNumericDouble )
      throw ParserError("expected numeric token not found","Parser::ReadSingleNumericToken") ;

   LoadNextToken() ;
}

//************************************************************

void Parser::ReadSingleTokenOpt
(
   TokenType type
)
{
   if ( tk.Type() == type )
      LoadNextToken() ;
}

//************************************************************

void Parser::ReadMeshVertexs()
{
   ReadSingleToken( tkBegin ) ;
   ReadSingleTokenOpt( tkVertexs ) ;

   while( tk.Type() == tkOpenPar ||
          tk.Type() == tkNormal  ||
          tk.Type() == tkIrad    ||
          tk.Type() == tkUV )
   {
      ReadVertex() ;
   }


   ReadSingleToken( tkEnd ) ;
   ReadSingleTokenOpt( tkVertexs ) ;
}

//************************************************************

void Parser::ReadMeshFaces()
{
   ReadSingleToken( tkBegin ) ;
   ReadSingleTokenOpt( tkFaces ) ;

   while( true )
   {
      if ( tk.Type() == tkNumericInteger )
         ReadFace() ;
      else if ( tk.Type() == tkOpenPar )
         ReadFace2() ;
      else if ( tk.Type() == tkRGB )
         ReadFacesColor() ;
      else if ( tk.Type() == tkTexture )
         ReadTextureUse() ;
      else
         break ;
   }

   ReadSingleToken( tkEnd ) ;
   ReadSingleTokenOpt( tkFaces ) ;
}

//************************************************************

void Parser::ReadFacesColor()
{
   ReadSingleToken(tkRGB) ;

   ReadVector3(currentFaceColorRGB) ;

   /**
   cout << std::endl << "** change face rgb: ("
       << currentFaceColorRGB[0] << ","
       << currentFaceColorRGB[1] << ","
       << currentFaceColorRGB[2] << ")" ;
   **/

   pCurrentFaceColorRGB = currentFaceColorRGB ;
}

//************************************************************

void Parser::ReadVertex()
{
   double vp[3] ;

   Vertex v ;

   if ( tk.Type() == tkNormal )
   {
      double vn[3] ;

      ReadSingleToken(tkNormal) ;
      ReadVector3(vn) ;

      if ( trStack != NULL )
      {
         Vector3d n(vn[0],vn[1],vn[2]) ;
         Vector3d n1 = trStack->transform.TrNormalVector(n) ;
         Vector3d n2 = n1.Normalized() ;

         vn[0] = n2(X) ;
         vn[1] = n2(Y) ;
         vn[2] = n2(Z) ;
      }

      v.SetNormal(vn) ;
   }

   if ( tk.Type() == tkIrad )
   {
      double irad[3] ;

      ReadSingleToken(tkIrad);
      ReadVector3(irad);
      v.SetIradiance(irad);
   }


   if ( tk.Type() == tkUV )
   {
      double uv[2] ;

      ReadSingleToken(tkUV);
      ReadVector2(uv) ;
      v.SetTextCoords(uv);
   }

   if ( currTextNum != -1 )
   if ( ! v.HasTextCoords() )
      throw ParserError("found vertex without texture coordinates in a textured mesh","Parser::ReadVertex");

   ReadVector3(vp) ;

   if ( trStack != NULL )
   {
      Vector3d v(vp[0],vp[1],vp[2]) ;

      v *= trStack->transform ;

      vp[0] = v(X) ;
      vp[1] = v(Y) ;
      vp[2] = v(Z) ;
   }

   v.SetPosition(vp) ;

   vertexs.push_back( v ) ;

   nVertexs++ ;
   nMeshVertexs ++ ;
}

//************************************************************


void Parser::ReadVector2
(
   double v[2]
)
{
   ReadSingleToken(tkOpenPar);

   v[0] = tk.ValueDouble() ;
   ReadSingleNumericToken();

   ReadSingleToken(tkComma) ;

   v[1] = tk.ValueDouble() ;
   ReadSingleNumericToken();

   ReadSingleToken(tkClosePar) ;
}

//************************************************************

void Parser::ReadVector3
(
   double v[3]
)
{
   ReadSingleToken(tkOpenPar) ;

   v[0] = tk.ValueDouble() ;
   ReadSingleNumericToken();

   ReadSingleToken(tkComma) ;

   v[1] = tk.ValueDouble() ;
   ReadSingleNumericToken();

   ReadSingleToken(tkComma) ;

   v[2] = tk.ValueDouble() ;
   ReadSingleNumericToken();

   ReadSingleToken(tkClosePar);
}

//************************************************************

void Parser::ReadFace()
{
   long nv = tk.ValueInteger() ;

   ReadSingleToken(tkNumericInteger) ;

   if ( nv < 0 || nv > 10000 )
      throw ParserError("invalid number of vertexs in face","ReadFace");

   std::vector<unsigned long> fv(nv) ;

   for( int i = 0 ; i < nv ; i++ )
   {
      long iv = tk.ValueInteger() ;

      if ( iv < 0L || nMeshVertexs <= (unsigned long)iv )
      {
         char s[200] ;
         sprintf(s,"vertex index out of bound (index=%ld,num.mesh.vertexs=%ld)",iv,nMeshVertexs) ;

         throw ParserError(s,"raised while reading vertex list in a face (Parser::ReadFace)") ;
      }

      fv[i] = (unsigned long)iv + meshVertexIndexBase ;

      ReadSingleToken(tkNumericInteger) ;
   }

   faces.push_back( Face(fv,currTextNum,pCurrentFaceColorRGB) ) ;
   nFaces ++ ;
}

//************************************************************

void Parser::ReadFace2()
{


   ReadSingleToken(tkOpenPar) ;

   std::vector<unsigned long> fv ;

   while( tk.Type() != tkClosePar )
   {
      long iv = tk.ValueInteger() ;

      ReadSingleToken(tkNumericInteger) ;

      if ( iv < 0L || nMeshVertexs <= (unsigned long)iv )
      {
         char s[200] ;
         sprintf(s,"vertex index out of bound ( index = %ld , num.mesh.vertexs = %ld )",iv,nMeshVertexs) ;

         throw ParserError(s,"raised while reading vertex list in a face (Parser::ReadFace2)") ;
      }

      fv.push_back(  (unsigned long)iv + meshVertexIndexBase ) ;

      if ( tk.Type() == tkComma )
         ReadSingleToken(tkComma) ;
      else
      if ( tk.Type() == tkClosePar )
         break ;
      else
      if ( tk.Type() != tkNumericInteger )
         throw ParserError("integer index expected in face","Parser::ReadFace2");
   }

   ReadSingleToken( tkClosePar ) ;

   faces.push_back( Face(fv,currTextNum,pCurrentFaceColorRGB) ) ;
   nFaces ++ ;
}

//************************************************************

Parser::~Parser()
{
   // no hay nada que borrar!

}

//************************************************************

const Face * Parser::GetFace
(
   unsigned long iFace
)
   const
{
   if ( iFace >=  nFaces )
   {
      char s[200] ;
      sprintf(s,"face index out of bounds (index=%ld,num.faces=%ld)",iFace,nFaces) ;

      throw ParserError(s,"Parser::GetFace") ;
   }

   return & ( faces[iFace] ) ;
}

//************************************************************

const Vertex * Parser::GetVertex
(
   unsigned long iVertex
)
   const
{

   if ( iVertex >=  nVertexs )
   {
     char s[200] ;
     sprintf(s,"vertex index out of bounds (index=%ld,num.vertexs=%ld)",iVertex,nVertexs) ;

     throw ParserError(s,"Parser::GetVertex") ;
   }

   return & ( vertexs[iVertex] ) ;
}

//*************************************************************

unsigned long Parser::NVertexs
()
   const
{
   return nVertexs ;
}

//*****************************************************************

unsigned long Parser::NFaces
()
   const
{
   return nFaces ;
}

//*************************************************************

const Light * Parser::GetLight
(
   unsigned long iLight
)
   const
{
   if ( iLight >=  nLights )
   {
     char s[200] ;
     sprintf(s,"light index out of bounds (index=%ld,num.lights=%ld)",
                  iLight,nLights) ;

     throw ParserError(s,"Parser::GetLight") ;
   }

   return &( lights[iLight] ) ;
}

//************************************************************

const Texture * Parser::GetTexture
(
   unsigned long iTexture
)
   const
{
   if ( iTexture >= nTextures )
   {
     char s[200] ;
     sprintf(s,"texture index out of bounds ( index = %ld, num.textures = %ld )",
                  iTexture,nTextures) ;

     throw ParserError(s,"Parser::GetTexure");
   }

   return &( textures[iTexture] ) ;
}

//***********************************************************

unsigned long Parser::NTextures
()
   const
{
   return nTextures ;
}

//***********************************************************

unsigned long Parser::NLights
()
   const
{
   return nLights ;
}

//********************************************************************************

void Parser::PushTransform
(
   const Transformd & tr
)
{
   Transformd newTransform ;

   if ( trStack != NULL )
      newTransform = trStack->transform ;

   newTransform = ( tr * newTransform ) ;

   TrStackNode * tmp = new TrStackNode ;

   if ( tmp == NULL )
      throw ParserError("unable to push transform","Parser::PushTransform");

   tmp->next      = trStack ;
   tmp->transform = newTransform ;

   trStack = tmp ;
}

//*******************************************************************************

void Parser::PopTransform()
{
   if ( trStack == NULL )
      throw ParserError("internal error: empty transform stack","Parser::PopTransform");

   TrStackNode * tmp = trStack ;

   trStack = trStack->next ;

   delete tmp ;
}

//**********************************************************************

void Parser::PushDefinitionsStackMark()
{
   DefStackNode * n = new DefStackNode(dfMark,"",NULL,defStack) ;

   if ( n == NULL )
      throw ParserError("unable to create object with 'new'","Parser::PushDefinitionsStackMark") ;

   defStack = n ;
}

//**************************************************************

void Parser::PopDefinitionsStack()
{

   DefinitionType type ;

   // "pop" from stack until a mark is deleted.

   do
   {
       if ( defStack == NULL )
           throw ParserError("internal error: cannot pop definitions stack: mark not found","Parser::PopDefinitionsStack");

       DefStackNode * n = defStack->next ;

       type = defStack->type ;

       DeleteDefStackNode(defStack) ;

       defStack = n ;
   }
   while( type != dfMark ) ;
}

//*************************************************************

void Parser::DeleteDefStackNode
(
   DefStackNode * n
)
{
   switch( n->type )
   {
      case dfMark :

         break ;

      case dfTexture :

         {
           unsigned long * p = (unsigned long *) n->data ;
           delete p ;
         }
         break ;

      default:

         throw ParserError("internal error: ilegal definition stack node type","Parser::DeleteDefStackNode");
   }

   delete n ;
}

//******************************************************

bool Parser::AnyVertexWithIradiance
()
   const
{
   for( unsigned long i=0 ; i < NVertexs() ; i++ )
      if ( GetVertex(i)->HasIradiance() )
             return true ;

   return false ;
}

//******************************************************

bool Parser::AnyVertexWithNormal
()
   const
{
   for( unsigned long i=0 ; i < NVertexs() ; i++ )
      if ( GetVertex(i)->HasNormal() )
             return true ;

   return false ;
}

//*****************************************************

void Parser::ComputeVertexNormals()
{

   std::cout << std::endl << "computing normals..." << std::flush ;

   double cero[3] = {0,0,0} ;

   for( unsigned long iv = 0 ; iv < NVertexs() ; iv ++ )
   {
          Vertex * v = & ( vertexs[iv] ) ;
          v->SetNormal( cero ) ;
   }

   for( unsigned long iface = 0 ; iface < nFaces ; iface ++ )
   {
       Face * f = & ( faces[iface] );
       unsigned nfv = f->NVertexs() ;

       if ( nfv < 3 )
          throw ParserError("found a face with less than 3 vertexs","Parser::ComputerVertexNormals") ;


       double p[3][3] ;

       Vector3d v[3] ;

       for( int i=0 ; i< 3 ; i++ )
       {
           GetVertex( f->GetVertexIndex(i) )->GetPosition( p[i] ) ;
           v[i] = Vector3d( p[i] ) ;
       }

       Vector3d faceNormal = ((v[1]-v[0])*(v[2]-v[0])).Normalized() ;

       for( unsigned iv = 0 ; iv < f->NVertexs() ; iv ++ )
       {
          Vertex * v = & ( vertexs[ f->GetVertexIndex(iv) ] ) ;

          double n[3] ;
          v->GetNormal(n) ;

          n[0] += faceNormal[X] ;
          n[1] += faceNormal[Y] ;
          n[2] += faceNormal[Z] ;

          v->SetNormal( n ) ;
       }

       for( unsigned iv = 0 ; iv < f->NVertexs() ; iv ++ )
       {
          Vertex * v = & ( vertexs[ f->GetVertexIndex(iv) ] ) ;

          double n[3] ;
          v->GetNormal(n) ;

          Vector3d normal = Vector3d( n ).Normalized() ;

          n[0] = normal[X] ;
          n[1] = normal[Y] ;
          n[2] = normal[Z] ;

          v->SetNormal( n ) ;
       }
   }

   std::cout << "done." << std::flush ;
}

//************************************************************************


void Parser::SaveAs
(
   const char * fileName
)
   const
{
   std::ofstream os(fileName) ;

   if ( ! os.good() )
   {
      std::string s = "cannot write onto file: " ;
      s += fileName ;
      throw ParserError(s,"Parser::SaveAs");
   }

   os << std::endl << "begin mesh"
      << std::endl << "   begin vertexs" ;

   for( unsigned long iv = 0 ; iv < NVertexs() ; iv++ )
   {
       const Vertex * v = GetVertex( iv ) ;

       if ( v->HasNormal() )
       {
          double n[3] ;
          v->GetNormal( n ) ;
          os << std::endl << "      normal ( " << n[0] << "," << n[1] << "," << n[2] << ")" ;
       }

       double p[3] ;
       v->GetPosition( p ) ;
       os << std::endl << "      ( " << p[0] << "," << p[1] << "," << p[2] << ")" ;
   }


   os << std::endl << "   end vertexs"
      << std::endl << "   begin faces"
      << std::endl << "      rgb (1,1,1)" ;

   for( unsigned long iff = 0 ; iff < NFaces() ; iff++ )
   {
      const Face * f = GetFace( iff ) ;

      for( unsigned i = 0 ; i < f->NVertexs() ; i++ )
      {
         unsigned long iv = f->GetVertexIndex( i ) ;

         if (i==0)
            os << std::endl << "      (" ;
         else
            os << "," ;

         os << iv ;
      }

      os << ")" ;
   }


   os << std::endl << "   end faces"
      << std::endl << "end mesh"
      << std::endl ;

   os.flush() ;

   os.close() ;
}


} ; // namespace GRF
