function [WeightsArray, isDependent] = ds(params, iBin, varargin)
% FUNCTION DESCRIPTION: 
%   Determines the coefficients of a delay&sum beamformer. 
% 
% INPUT PARAMETERS: 
%   (struct) params.Ma.Beam.Sd             ... steering direction 
%            params.Ma.Basic.c             ... sound velocity
%            params.Ma.Array.nMicTotal     ... number of microphones 
%            params.Ma.Array.Mic.rArray    ... microphone radius 
%            params.Ma.Array.Mic.AzimArray ... microphone azimuth
%            params.Ma.Array.Mic.ElevArray ... microphone elevation 
%            params.Global.Audio.fArray    ... frequency 
%   (int)    iBin                          ... bin number 
% 
% OUTPUT PARAMETERS: 
%   (array)  WeightsArray ... calculated weights 
%   (bool)   isDependent  ... information about data-dependency of beamformer 
%
% UPDATE: 09. October 2012, Hannes Pessentheiner 

%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
%  HEADER:   
%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
if nargin < 2
    error('ERROR: Not enough input parameters. Function terminated.');
end

%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
%  DEFINITIONS:   
%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
OPT_TORAD    = pi/180; 
isDependent  = 0; % data-independent beamformer 

WeightsArray = zeros(params.Ma.Array.nMicTotal, 1, 'single');   

%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
%  PROCESS:   
%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
% Vector (Lx1)
SinCosArray = sin(params.Ma.Beam.Sd(2)                                   * OPT_TORAD) ...  
           .* sin(params.Ma.Array.Mic.ElevArray                          * OPT_TORAD) ... % vec (Lx1)
           .* cos((params.Ma.Beam.Sd(1) - params.Ma.Array.Mic.AzimArray) * OPT_TORAD) ... % vec (Lx1)
            + cos(params.Ma.Beam.Sd(2)                                   * OPT_TORAD) ... 
            * cos(params.Ma.Array.Mic.ElevArray                          * OPT_TORAD);    % vec (Lx1)
% Vector (Lx1)
FrequencyIndependentDelayArray                                                        ...
            = params.Ma.Array.Mic.rArray                                              ... % vec (Lx1)
            / params.Ma.Basic.c                                                       ... 
           .* SinCosArray;                                                                % vec (Lx1)
% Vector (Lx1)
FrequencyDependentDelayArray                                                          ...
            = -1i                                                                     ...
            * 2                                                                       ...
            * pi                                                                      ...
            * params.Global.Audio.fArray(iBin)                                        ...
            * FrequencyIndependentDelayArray;                                             % vec (Lx1)
% Vector (Lx1)
WeightsArray(:)                                                                       ...
            = 1 / params.Ma.Array.nMicTotal                                           ...
           .* exp(FrequencyDependentDelayArray);  
end

%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
%  INFO:   
%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
% MATRIX/VECTOR DIMENSIONS: 
%   K ... number of frequency bins 
%   L ... number of microphones 
%   M ... number of elevation angles 
%   N ... number of azimuthal angles 