function [DfrArray] = computeDesiredFrequencyResponse(nBin, AudioOutFs) 
% FUNCTION DESCRIPTION: 
%   LP-filters the output signal. Computes the desired 
%   frequency response to remove DC components and low-frequency noise, 
%   which is usually introduced by the power net. Removes the signals 
%   above a certain frequency to mitigate the imperfect anti-aliasing 
%   filter that precedes the ADC. Output in bins, not in frequencies.
%
% INPUT PARAMETERS: 
%   (int)    nBin       ... number of bins   
%   (int)    AudioOutFs ... sampling frequency
% 
% OUTPUT PARAMETERS: 
%   (array)  DfrArray ... desired frequency response 
% 
% The frequency response looks as follows: 
%   F(f) =  0                   , f       <  fmin
%           0.5*(1+cos(x)/x)    , fmin    <= f <= fstart
%           1                   , fstart  <= f <= fend 
%           0.5*(1+cos(y)/y)    , fend    <= f <= fmax
%           0                   , f       >  fmax
% 
% UPDATE: 09. October 2012, Hannes Pessentheiner 

%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
%  HEADER: 
%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
if nargin < 2
    error('ERROR: Not enough input parameters. Function terminated.');
end

%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
%  DEFINITIONS: 
%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
DfrArray = zeros(nBin/2, 1, 'single'); 

%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
%  PROCESS: 
%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
% Define start/end/cut-off frequencies. 
BinStart     = floor( 100               / AudioOutFs * nBin); % zero response below this frequency 
BinStartFull = ceil ( 200               / AudioOutFs * nBin); % unit gain after this frequency
BinEndFull   = floor((AudioOutFs/2-100) / AudioOutFs * nBin); % unit gain up to this frequency 
BinEnd       = floor( AudioOutFs/2      / AudioOutFs * nBin); % zero response after this frequency 

% Calculate frequency response mentioned above: 
DfrArray(1:BinStart) = 0;

for iBin = BinStart:BinStartFull
    DfrArray(iBin) = 0.5 * (1.0 + cos((iBin - BinStartFull) / (BinStartFull - BinStart)*pi));
end

DfrArray(BinStartFull:BinEndFull) = 1;
for iBin = BinEndFull:BinEnd
    DfrArray(iBin) = 0.5 * (1.0 + cos((iBin - BinEndFull) / (BinEnd - BinEndFull)*pi));
end

end