function xest= bvts( data, nest, nvar, gmm_x, imputationAlgorithm, vtsOrder )

%--------------------------------------------------
%Definition of anonymous functions for the function
%--------------------------------------------------

g0= @(x, n) log( 1 + exp(n-x) );
f0= @(x, n) 1./(1+exp(n-x));

%--------------------------------------------------
% Function definition
%--------------------------------------------------

switch imputationAlgorithm
    case 'simple'
        estimate= @vtsSimpleEstimationFast;
    case 'map'
        estimate= @vtsMapEstimationFast;
    case 'bounded'
        estimate= @vtsBoundedEstimationFast;
end

xest= data;
[params frames]= size(data);
K= gmm_x.ncentres;
p= zeros(K, 1);
for t= 1:frames
    y= data(:,t);
    n= nest(:,t);
    sn= nvar(:,t);

    %Precompute terms used in VTS adaptation for speed
    G0= zeros(params, K);
    F0= zeros(params, K);
    for k= 1:K
        G0(:,k)= g0( gmm_x.centres(k,:)', n );
        F0(:,k)= f0( gmm_x.centres(k,:)', n );
    end
    gmm_y= adaptGMMFast( gmm_x, n, sn, vtsOrder, G0, F0 );

    %Compute the posteriors for every Gaussian
    for k= 1:K
        p(k)= probability(gmm_y, k, y) * gmm_y.priors(k);
    end
    p= p/sum(p);
    
    %Compute the partial estimates
    Mest= zeros(params, K);
    for k= 1:K
        Mest(:,k)= estimate( gmm_x, gmm_y, k, n, sn, y, G0(:,k), F0(:,k) );
    end
    
    %Combine the partial estimates according to their posterior
    %probabilities
    xest(:,t)= sum(bsxfun(@times, Mest, p'), 2);     
end


%-------------------------------------------------------------------------
function gmm_y= adaptGMMFast( gmm_x, nest, nvar, order, G0, F0 )

forceSymmetric= @(X) triu(X) + triu(X, 1)';  %Anonymous function definition
gmm_y= gmm_x;

for k= 1:gmm_x.ncentres
    %Mean adaptation
    mx= gmm_x.centres(k,:)';
    gmm_y.centres(k,:)= (mx + G0(:,k))';
    
    %Variance adaptation
    if order > 0
        if strcmp(gmm_x.covar_type,'diag')
            sx= gmm_x.covars(k,:)';
            gmm_y.covars(k,:)= (F0(:,k).^2 .* sx + (1-F0(:,k)).^2 .* nvar)';
        else
            Sx= gmm_x.covars(:,:,k);
            gmm_y.covars(:,:,k)= forceSymmetric( diag(F0(:,k))*Sx*diag(F0(:,k)) + diag((1-F0(:,k)).^2 .* nvar) );
        end
    end
end


%--------------------------------------------------------------------------
function p= probability( gmm, k, x )

% if isempty( gmm )
%     p= 1;
% else
mean= gmm.centres(k,:)';
if strcmp(gmm.covar_type,'diag')
    s= 1./sqrt(gmm.covars(k,:)');
    z= s.*(x-mean);
    p= prod( normpdf(z).*s );
else
    S= gmm.covars(:,:,k);
    p= mvnpdf( x, mean, S );
end
% end


%--------------------------------------------------------------------------
function xest= vtsSimpleEstimationFast( gmm_x, gmm_y, k, nest, nvar, y, g0, f0 )
xest= y - g0;


%--------------------------------------------------------------------------
function varargout= vtsMapEstimationFast ( gmm_x, gmm_y, k, nest, nvar, y, g0, f0 )

mx= gmm_x.centres(k,:)';
my= gmm_y.centres(k,:)';

if strcmp(gmm_x.covar_type,'diag')
    sx= gmm_x.covars(k,:)';
    sy= gmm_y.covars(k,:)';
    sxy= sx .* f0;
    z= (y-my)./sy;
    m_x_y= mx + sxy.*z;
    S_x_y= diag( sx - sxy.^2./sy );
else
    Sx= gmm_x.covars(:,:,k);
    Sy= gmm_y.covars(:,:,k);
    Sxy= Sx*diag(f0);
    A= Sxy/Sy;
    m_x_y= mx + A*(y-my);
    S_x_y= Sx - A*Sxy';
end

xest= m_x_y;
varargout{1}= xest;
if nargout > 1
    varargout{2}= m_x_y;
    varargout{3}= S_x_y;
end


%--------------------------------------------------------------------------
function xest= vtsBoundedEstimationFast ( gmm_x, gmm_y, k, nest, nvar, y, g0, f0 )

%Min values for every log-Mel filterbank obtained from the Aurora2 training dataset
LOWER_BOUND_A2= [0 0 0 0 0 0 0.4741 0.6503 0.7766 0.8355 1.2978 1.6634 1.8870 1.8651 1.7529 2.4891 2.7295 2.7208 3.0669 3.1605 3.2724 3.5133 3.1771]';

[xest, m_x_y, S_x_y]= vtsMapEstimationFast( gmm_x, gmm_y, k, nest, nvar, y, g0, f0 );

s_x_y= sqrt(abs(diag(S_x_y)));
up= (y-m_x_y)./s_x_y;
low= (LOWER_BOUND_A2-m_x_y)./s_x_y;
q= s_x_y.*(normpdf(low)-normpdf(up))./(normcdf(up)-normcdf(low));
q(isnan(q) | isinf(q))= 0;
m_x_y2= m_x_y + q;

xest= max(LOWER_BOUND_A2, min(m_x_y2, y));
