function [status] = MatJobSub(job,kind)
% submits a job to the batch system, the parameter job is an array containing
% information on the job to be submitted. If job is not set, then default
% parameters are used.
% job = struct('name', 'SomeJob');                          ... name of job
% job.email = 'foobar@tugraz.at';                           ... status email addressee
% job.emailJobNone = false;                                 ... no status emails
% job.emailJobBegin = false;                                ... email on start of job
% job.emailJobEnd = false;                                  ... email on end of job
% job.emailJobAbort = true;                                 ... email on abort of job
% job.emailJobSuspended = false;                            ... email on suspended job
% job.errorFile = '/clusterFS/foobar/err/job.err';          ... error file
% job.outputFile = '/clusterFS/foobar/out/job.out';         ... std. out file
% job.workingDirectory = '/clusterFS/foobar';               ... working path
% job.dependencies = {};                                    ... dependencies (e.g. {{842},'quad'})
% job.hardlimit.h_vmem = '2G';                              ... hardlimits (e.g. hardlimit 'h_vmam' is set to '2G')
% job.softlimit.legen = 'dary';                             ... softlimits (e.g. softlimit 'legen' is set to 'dary')
% job.restart = true;                                       ... restart aborted job
% job.slots = 1;                                            ... number of slots required to run this job
% job.queue = '';                                           ... define the simulation queue (e.g. 'main.q')
%
% MATLAB:
% job.type = 'matlab';                                      ... job type 'matlab'
% job.command = 'coolMfun(''/clusterFS/foobar/data.dat'')'; ... matlab function with input arguments
%
% OCTAVE:
% job.type = 'octave';                                      ... job type 'octave'
% job.command = 'coolOfun(''/clusterFS/foobar/data.dat'')'; ... octave function with input arguments
%
% ADS:
% job.type = 'ads';                                         ... job type 'ads'
% job.command = 'coolADSnetlist.log';                       ... ADS netlist
%
% Shell:
% job.type = 'shell';                                       ... job type 'shell'
% job.command = 'testshellscript.sh';                       ... shell command
%
% For further information on job types, dependencies, hard- and softlimits have a look into the Grid Engine User Guide.
%
% Changelog:
% * Spring 2012: Sebastian Tschiatschek
%    - initial version for running 'matlab' and 'octave' jobs
%    - adding job dependencies
% * June 2012: Hannes Pessentheiner
%    - adding 'compiled' jobs
% * June 2012: Sebastian Tschiatschek
%    - adding 'shell' jobs
% * 24 June 2012: Hannes Pessentheiner & Markus Koeberl
%    - adding hard- and softlimits
% * 27 June 2012: Hannes Pessentheiner
%    - adding 'jobpathname' into script output status
% * 13 July 2012: Andreas Pedross
%    - adding 'ads' jobs
%    - improved readability of code
%    - commenting full code
% * 18 Sept. 2012: Sebastian Tschiatschek
%    - adding 'slots'
% * 16 Oct. 2012: Andreas Pedross
%    - removing 'compiled' job (obsolete, same as 'shell'!)
%    - merging 'submitjob.m'-braches (Tschiatschek & Pedross/Pessentheiner)
%    - commenting code and fixing help
% * 09 Nov. 2012: Sebastian Tschiatschek
%    - added error massage for Octave jobs
% * 30 April 2013: Andreas Pedross & Markus Koeberl
%    - new softlimit 'matlab_${user}=1' for matlab jobs for better licence usage
% * 06 June 2013: Andreas Pedross
%    - minor change for checking supported jobs


%% Set User Name and Path
[tmp, username] = system('whoami');
username = strtrim(username);

homedir = ['/clusterFS/home/user/', username, '/'];

%% Checking and Generate Job Structure

% check if there exists a job
if nargin == 0 || isempty(job)
    job = struct('name', sprintf('%s''s job', username));
end

% check if job is a structure
if ~isstruct(job)
    error('Passed job descriptor is not a structure.');
end

% check if job does have a name
if ~isfield(job, 'name')
    job.name = sprintf('%s''s job', username);
end

% check if jobs does have a email address, if not, search LDAP search
if ~isfield(job, 'email')
    [tmp,email] = system(sprintf('ldapsearch -x uid=%s mail | awk ''/^mail:/ {print $2}''', username));
    email = strtrim(email);
    job = setfield(job, 'email', email);
end

% if no email option is set, send email on error
if ~isfield(job, 'emailJobNone') && ~isfield(job, 'emailJobBegin') && ~isfield(job, 'emailJobEnd') && ~isfield(job, 'emailJobAbort') && ~isfield(job, 'emailJobSuspended'), 
    job = setfield(job, 'emailJobAbort', true);
end

% set email default values if not existing
if ~isfield(job, 'emailJobNone')
    job = setfield(job, 'emailJobNone', false);
end

% set email default values if not existing
if ~isfield(job, 'emailJobBegin')
    job = setfield(job, 'emailJobBegin', false);
    job.emailJobNone = false;
end

% set email default values if not existing
if ~isfield(job, 'emailJobEnd')
    job = setfield(job, 'emailJobEnd', false);
    job.emailJobNone = false;
end

% set email default values if not existing
if ~isfield(job, 'emailJobAbort')
    job = setfield(job, 'emailJobAbort', true);
    job.emailJobNone = false;
end

% set email default values if not existing
if ~isfield(job, 'emailJobSuspended')
    job = setfield(job, 'emailJobSuspended', false);
    job.emailJobNone = false;
end

% set path of error file if not existing
if ~isfield(job,'errorFile')
    job = setfield(job, 'errorFile', [homedir, 'err/']);
end

% set path of standard out file if not existing
if ~isfield(job,'outputFile')
    job = setfield(job, 'outputFile', [homedir, 'out/']);
end

% set working directory if not existing
if ~isfield(job,'workingDirectory')
    job.workingDirectory = homedir;
end

% check if job type does exist
if ~isfield(job, 'type')
    job.type = 'none';
end

% check if there exists a command
if ~isfield(job, 'command')
    error('No command for the job specified.');
end

% check dependencies
if ~isfield(job, 'dependencies')
    job = setfield(job, 'dependencies', {});
end

% check if a aborted job should be restarted
if ~isfield(job, 'restart')
    job.restart = true;
end

% check if there exist any limits
if ~isfield(job,'hardlimit') && ~isfield(job,'softlimit')
    job.hardlimit.h_vmem = '2G';
end

% by default, assume that a job needs only one slot
if ~isfield(job, 'slots'),
    job.slots = 1;
end

%% Generate Job Script

jobfile = tempname();
fid = fopen(jobfile, 'w');

% make bash script and setting parameters
fprintf(fid, '#!/bin/bash\n');
fprintf(fid, '#\n');
fprintf(fid, '#$ -N "%s"\n', job.name);
fprintf(fid, '#$ -wd %s\n', job.workingDirectory);
fprintf(fid, '#$ -o %s\n', job.outputFile);
fprintf(fid, '#$ -e %s\n', job.errorFile);
fprintf(fid, '#$ -S /bin/bash\n');
fprintf(fid, '#$ -M %s\n', job.email);

% setting softlimits
if isfield(job, 'softlimit')
    limits = getfield(job, 'softlimit');
    for idx_limit=1:numel(fields(limits))
        tempvar = fieldnames(limits);
        for idx_tempvar=1:numel(tempvar) 
            fprintf(fid, '#$ -soft -l %s=%s\n', char(tempvar(idx_tempvar)), getfield(limits, char(tempvar(idx_tempvar))));
        end
    end
end
if strcmp(job.type, 'matlab')
    fprintf(fid, '#$ -soft -l matlab_%s=1\n', username);
end

% setting hardlimits
if isfield(job, 'hardlimit')
    limits = getfield(job, 'hardlimit');
    for idx_limit=1:numel(fields(limits))
        tempvar = fieldnames(limits);
        for idx_tempvar=1:numel(tempvar) 
            fprintf(fid, '#$ -hard -l %s=%s\n', char(tempvar(idx_tempvar)), getfield(limits, char(tempvar(idx_tempvar))));
        end
    end
end

% setting queues
if isfield(job, 'queue') && ~strcmp(job.queue, ''),
    fprintf(fid, '#$ -q %s\n', job.queue);
end

% setting slots
if job.slots > 1,
  fprintf(fid, '#$ -pe serial %d\n', job.slots);
end

% set email parameters
emailstr = '';

if job.emailJobNone
    emailstr = [emailstr, 'n'];
end

if job.emailJobBegin
    emailstr = [emailstr, 'b'];
end

if job.emailJobEnd
    emailstr = [emailstr, 'e'];
end

if job.emailJobAbort
    emailstr = [emailstr, 'a'];
end
if job.emailJobSuspended
    emailstr = [emailstr, 's'];
end

fprintf(fid, '#$ -m %s\n', emailstr);

% set dependencies
if ~isempty(job.dependencies)
    if ischar(job.dependencies(1))
        depstr = job.dependencies{1};
    else
        depstr = int2str(job.dependencies{1});
    end
    for i=2:length(job.dependencies),
        if ischar(job.dependencies{i})
            depstr = [depstr, ',', job.dependencies{i}];
        else
            depstr = [depstr, ',', int2str(job.dependencies{i})];
        end
    end
    fprintf(fid, '#$ -hold_jid %s\n', depstr);
end

% set job restart settings 
if job.restart == true,
    restart = 'yes';
else
    restart = 'no';
end
fprintf(fid, '#$ -r %s\n', restart);

fprintf(fid, '\n');

% set home path
fprintf(fid, 'HOME=%s\n',homedir);

% set job commands
if strcmp(job.type, 'matlab')
    % Matlab Job:
    fprintf(fid, 'MATLAB=/afs/spsc.tugraz.at/opt/matlab/R2012a/ /usr/local/bin/matlabR2012a.afs -nodisplay -nosplash -singleCompThread -r "try, %s; catch ME, disp(ME); exit(-1); end; exit(0);"', strrep(job.command, '"', '\"'));    
    if (strcmp(kind,'s'))
    fprintf(fid, 'MATLAB=/afs/spsc.tugraz.at/opt/matlab/R2012a/ /usr/local/bin/matlabR2012a.afs -nodisplay -nosplash -singleCompThread -r "try, %s; %s; catch ME, disp(ME); exit(-1); end; exit(0);"', ['cd ' job.workingDirectory], strrep(job.command, '"', '\"'));    
    end

elseif strcmp(job.type, 'octave')
    % Octave Job
    fprintf(fid, '/afs/spsc.tugraz.at/opt/octave/3.6.2/bin/octave --no-window-system --eval "try, %s; catch msg=lasterror(); disp(msg.message); for [val,key] = msg.stack, key, val, end, exit(-1); end; exit(0);"', strrep(job.command, '"', '\"'));
elseif strcmp(job.type, 'ads')
    % ADS jobs
    fprintf(fid, '/usr/local/bin/hpeesofsim.afs %s', job.command);
elseif strcmp(job.type, 'shell')
    % Shell command
    fprintf(fid, '%s', job.command);
    warning('This job type has not been tested yet.');
else
    % Job not supported!?
    error('Job type is not supported. Supported job types are: %s\n',[jobtypes{:}]);
end

fprintf(fid, '\n');
fclose(fid);

status = struct('jobfile', jobfile);

%% Submit Job

% set home environment for the script call
HOME = getenv('HOME');
setenv('HOME', homedir);

% submit job to the Grid Engine
%system(['cat ' jobfile]);
fprintf('Submitted %s\n',jobfile);

switch kind
    case 's'
        system(['sh ' jobfile]);
        a=0; b='Your job 1126024 ("jamcs job") has been submitted';
    case 'p'
        [a,b] = system(sprintf('qsub %s', jobfile));      
end



if a ~= 0
    disp(jobfile);
    error('Job submission failed.');
end

% set home environment for the script call back
setenv('HOME', HOME);

% set job status at submission
x = regexp(b, 'Your job (\d+)*+', 'tokens');
if iscell(x{1}),
   x = x{1};
end   
status = setfield(status, 'jobid', str2num(x{1}));
status = setfield(status, 'message', strtrim(b));

end
