#include <stdlib.h> //rand, abs, system 
#include <stdio.h> //fopen, printf 
#include <math.h> //sqrt, fabs, ceil
#include "./Math.h"  //PIx2


/////////////////////////////////////////////////////////////////////// CORRELATIONS ///////////////////////////////////////////////////
/*Cross Correlation Biased
lx: length x*/
int xcorr (float *Corr, float *x1, float *x2, int lx, char kind)
{
	int i, s;
	float w[2*lx-1+2], aux;
	
	//Pos part
	for(s=0; s<lx; s++)
	{
		aux=0;
		for(i=0; i<lx-s; i++)
			aux=aux+(x1[i]*x2[i+s]);
		Corr[s+lx-1]=aux/(float)(lx-s);
	}

	//Neg part
	for(s=0; s<lx; s++)
	{
		aux=0;
		for(i=0; i<lx-s; i++)
			aux=aux+(x1[i+s]*x2[i]);
		Corr[lx-s-1]=aux/(float)(lx-s);
	}

	//If Biased
	if(kind=='b')
	{
		TriangWin (w, 2*lx-1+2); //+2 because Corr[0] with w[1] is not 0 (Barlett).
		for(i=0; i<2*lx-1; i++)
			Corr[i]=Corr[i]*w[i+1];		
	}	
	
	return 2*lx-1;
		
}




/**********************************
* 	Rx OSA with biased
*	
*	EXAMPLE
*	If x={1,2,3,4} , and FL=4
*	returns 7.500000 5.000000 2.750000 1.000000
*	= R(0), R(1), R(2), R(3), the zero and positive part of Autocorr biased
*	Obtain the same result xcorr('biased') of Matlab
****************************************/
void rxosa (float *ACorr, float *x, int tam, char kind) 
{
	int i, s;
	float aux;
	
	if(kind=='b')
	{
		for (s=0; s<tam; s++)
		{
			aux=0;
			for (i=0; i<tam-s; i++)
				aux=aux+(x[i]*x[i+s]);
			ACorr[s]=aux/(float)(tam);
		}
	}
	if(kind=='u')
	{
		for (s=0; s<tam; s++)
		{
			aux=0;
			for (i=0; i<tam-s; i++)
				aux=aux+(x[i]*x[i+s]);
			ACorr[s]=aux/(float)(tam-s);
		}
	}

	



}


/*
	CONVOLUTION c(w)=sum[x(th)*h(w-th)]
	x:	x(0),...x(L-1)
	h:	h(-(L-1)),...h(0),...., h(L-1),
	c:	c(0),...., c(L-1)
*/
int conv(float *c ,float *x, float *h, int L)
{
	int w, th;
	
	for(w=0; w<L; w++)
	{	
		c[w]=0;
		for(th=0; th<L; th++)
			c[w]+=x[th]*h[(w+L-1)-th];
	}

	return L;
}

/* Circular convolution */
int cconv (float *c, float *x, float *y, int L)
{
	int i,j;
	float yr[L], cm[L][L];

	
	//Reflection of y
	yr[0]=y[0];
	for(i=L-1; i>0; i--)
		yr[L-i]=y[i];
	
	//Circular Matrix
	for(i=0; i<L; i++)
	{	
		for(j=0; j<L; j++)
			cm[i][j]=yr[(i+j)%L];
	}	
	
		
	

	//Multiplication
	MultMV(c, &cm[0][0], x, L);
	
	
	
	return L;


}


//////////////////////////////////////////////////////////// LPC ///////////////////////////////////////////////////////////////////////

/*LPC by LEVISON-DURBIN
rx:	.....rx(-1) r(0) r(1),....
ac:	ac=[1 a(1) a(2) ...a(p)]
x(n)=-a(1)x(n-1)-a(2)x(n-2)-a(3)x(n-3)...+e(n)
*/
float levinson(float *ac, float *rx, int lrx, int p)
{
	int lx, i, j;
	float k, E, aux, var, apr[p+1];
	
	lx=(lrx+1)/2;
	
	E=rx[lx-1];
	
	for(i=1; i<=p; i++)
	{
		//k obtain
		aux=rx[i+lx-1];	
		for(j=1; j<i; j++)
			aux-=apr[j]*rx[i-j+lx-1];
		k=aux/E;
		
		//ac obtain
		ac[i]=k;    
		for(j=1; j<i; j++)
			ac[j]=apr[j]-k*apr[i-j];
		E=(1-k*k)*E;
    
		//Update
		for(j=1; j<=p; j++)
			apr[j]=ac[j];	
	}
		
	//Output
	ac[0]=1;
	for(j=1; j<=p; j++)
			ac[j]=-ac[j];
	
	var=0;
	for(j=0; j<=p; j++)
		var+=ac[j]*rx[lx-1+j];
	
	return var;
}


///////////////////////////////////////////////////////// FFT //////////////////////////////////////////////////////////////////////////////

/*---------------------------------------------------------------------------
 * FUNCTION NAME: rfft
 *  
 * PURPOSE:       Real valued, in-place split-radix FFT
 * 
*  void  rfft (float *x, int n, int m)
 * INPUT:
 *   x             Pointer to input and output array
 *   n             Length of FFT, must be power of 2
 *  m		must be log2(m)
 * OUTPUT         Output order
 *                  Re(0), Re(1), ..., Re(n/2), Im(n/2-1), ..., Im(1)
 *
 *  EXAMPLE 1:
 *   If x={1, 2, 3, 4, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0} and n=16, m=4
 *  return fft of {1,2,3,4} with 16 points (but only 8 coeffs). The 0s are where fft 
 *  puts its output
 *  EXAMPLE 2:
 *  If x={1, 2, 3, 4, 5, 6, 7, 8, 1, 2, 3, 4, 5, 6, 7, 8}; and n=4, m=2 
 *  return fft of x with 4 points, also x is truncated and algorith only use the
 *  first 4 points of x
  *---------------------------------------------------------------------------*/
void 
rfft (float *x, int n, int m)
{
    int j, i, k, is, id;
    int i0, i1, i2, i3, i4, i5, i6, i7, i8;
    int n2, n4, n8;
    float xt, a0, e, a, a3;
    float t1, t2, t3, t4, t5, t6;
    float cc1, ss1, cc3, ss3;
    float *r0;

    /* Digit reverse counter */

    j = 0;
    r0 = x;

    for (i = 0; i < n - 1; i++)
    {

        if (i < j)
        {
            xt = x[j];
            x[j] = *r0;
            *r0 = xt;
        }
        r0++;

        k = n >> 1;

        while (k <= j)
        {
            j = j - k;
            k >>= 1;
        }
        j += k;
    }

    /* Length two butterflies */
    is = 0;
    id = 4;

    while (is < n - 1)
    {

        for (i0 = is; i0 < n; i0 += id)
        {
            i1 = i0 + 1;
            a0 = x[i0];
            x[i0] += x[i1];
            x[i1] = a0 - x[i1];
        }

        is = (id << 1) - 2;
        id <<= 2;
    }

    /* L shaped butterflies */
    n2 = 2;
    for (k = 1; k < m; k++)
    {
        n2 <<= 1;
        n4 = n2 >> 2;
        n8 = n2 >> 3;
        e = (M_PI * 2) / n2;
        is = 0;
        id = n2 << 1;
        while (is < n)
        {
            for (i = is; i <= n - 1; i += id)
            {
                i1 = i;
                i2 = i1 + n4;
                i3 = i2 + n4;
                i4 = i3 + n4;
                t1 = x[i4] + x[i3];
                x[i4] -= x[i3];
                x[i3] = x[i1] - t1;
                x[i1] += t1;

                if (n4 != 1)
                {
                    i1 += n8;
                    i2 += n8;
                    i3 += n8;
                    i4 += n8;
                    t1 = (x[i3] + x[i4]) / M_SQRT2;
                    t2 = (x[i3] - x[i4]) / M_SQRT2;
                    x[i4] = x[i2] - t1;
                    x[i3] = -x[i2] - t1;
                    x[i2] = x[i1] - t2;
                    x[i1] = x[i1] + t2;
                }
            }
            is = (id << 1) - n2;
            id <<= 2;
        }

        for (j = 1; j < n8; j++)
        {
            a = j * e;
            a3 = 3 * a;
            cc1 = cos (a);
            ss1 = sin (a);
            cc3 = cos (a3);
            ss3 = sin (a3);

            is = 0;
            id = n2 << 1;

            while (is < n)
            {
                for (i = is; i <= n - 1; i += id)
                {
                    i1 = i + j;
                    i2 = i1 + n4;
                    i3 = i2 + n4;
                    i4 = i3 + n4;
                    i5 = i + n4 - j;
                    i6 = i5 + n4;
                    i7 = i6 + n4;
                    i8 = i7 + n4;
                    t1 = x[i3] * cc1 + x[i7] * ss1;
                    t2 = x[i7] * cc1 - x[i3] * ss1;
                    t3 = x[i4] * cc3 + x[i8] * ss3;
                    t4 = x[i8] * cc3 - x[i4] * ss3;
                    t5 = t1 + t3;
                    t6 = t2 + t4;
                    t3 = t1 - t3;
                    t4 = t2 - t4;
                    t2 = x[i6] + t6;
                    x[i3] = t6 - x[i6];
                    x[i8] = t2;
                    t2 = x[i2] - t3;
                    x[i7] = -x[i2] - t3;
                    x[i4] = t2;
                    t1 = x[i1] + t5;
                    x[i6] = x[i1] - t5;
                    x[i1] = t1;
                    t1 = x[i5] + t4;
                    x[i5] = x[i5] - t4;
                    x[i2] = t1;
                }
                is = (id << 1) - n2;
                id <<= 2;
            }
        }
    }
}



/*
MAGNITUDE OF DFT (output is equivalent to Mfft)
M: 		Out magnitude of dft. M(0), M(1), M(ndft-1)
x:		In signal
lx:		Length of x in 
ndft: 		n of point from 0 to 2*pi of dft, usually equal to p (pitch)
return		The final length of M
*/
int MDft (float *M, float *x, int ndft, int lx) 
{
	int k, n;
	float R[ndft], I[ndft], wk[ndft], a1, a2;	
	

	//set of wk
	for(k=0; k<ndft; k++)
		wk[k]=((float)k*PIx2)/(float)ndft;
		
	
	//M(0), M(1), .., M(ndfft-1)
	for(k=0; k<ndft; k++)
	{		
		//Cos and Sin part
		a1=0; a2=0;
		for(n=0; n<lx; n++)
		{
			a1+=x[n]*cos((n-lx/2)*wk[k]);					
			a2+=x[n]*sin((n-lx/2)*wk[k]);		
		}
		R[k]=a1;		
		I[k]=a2;
		M[k] = (float)sqrt((double)(a1*a1+a2*a2));		
	}

	return ndft;
}

/*Magnitude Spectrum of Harmonic Signal  
T: pitch
x: harmonic signal
N:  n of points of Spetrum from 0 to 2*PI (usually SP 128)
*/
void MHarm(float *Sp, float *x, int lx, int N, int p)
{
	int i, pos;
	float Harm[p], b0;	
	
	
	MDft(Harm, x, p, lx);	
	
	
	
	Zeros(Sp, N);
	b0=(float)N/(float)p;
	
	for(i=0; i<p; i++)
	{
		pos=float2int(b0*(float)i);
		
		Sp[pos]=Harm[i];
	}			
	
}

/*	Magnitude Spectrum using fft
	Sp:	Out with Magnitude spectrum, Sp(0),....., Sp(2pi-1), Sp[0],.....Sp[N-1]
	x:	In signal
	w:	window
	N:	n of complex points of fft from 0 to 2*PI (must be power of 2)
	log2N:	log2 of N
	lx:	length of x and w	
*/
void Mfft (float *Sp, float *x, float *w, int lx, int N, int log2N)
{
	int i;	
	float a[2*N];
	
	//Windowed
	for(i=0; i<lx; i++)
		a[i]=w[i]*x[i];
		
	
	
	// Zero padding 
	for(i=lx; i<N; i++)
		a[i] = 0.0;
	
	
	
	
	//FFT
	rfft(a, N, log2N); //In: a = x (with pad 0s), Out: Re(0),... Re(N/2), Im(N/2-1),... Im(1)
	
	//ABS
	a[0] = (float)fabs((double)a[0]);  // DC 
	for (i = 1; i < N / 2; i++)  //pi/(N/2), 2pi/(N/2), (N/2-1)*pi/(N/2) 
		a[i] = (float) sqrt((double)a[i] * (double)a[i] + (double)a[N-i] * (double)a[N-i]);	
	a[N/2] = (float)fabs((double)a[N/2]);  //pi
	


	//Repeat
	Sp[0]=a[0];
	for(i=1; i<=N/2; i++)  
	{	
		Sp[i]=a[i];
		Sp[N-i]=a[i];
	}


	
}




/*
Sp:	Sp(0)....Sp(2pi)
Nfft: 	Number of fft points,  from 0 to 2pi
p: 	log2(Nfft) ('x' and 'r'), pitch ('h'), LPC order ('l')
*/
void PSD (float *Sp, float *x, float *w, int Nfft, int p, int lx, char kind)
{
	int i;
	float var, ac[p+1];
	
	switch(kind)
	{
		case 'x':
			Mfft (Sp, x, w, lx, Nfft, p);
			for(i=0; i<Nfft; i++)
				//Sp[i]=Sp[i]*Sp[i]/(float)lx;	
				Sp[i]=Sp[i];
			break;
			
		case 'h':
			MHarm(Sp, x, lx, Nfft, p);
			
			for(i=0; i<Nfft; i++)
				Sp[i]=Sp[i]/(float)lx;
				//Sp[i]=Sp[i]*Sp[i]/(float)lx;	
			break;			
		
		case 'r':
			Mfft (Sp, x, w, lx, Nfft, p);
			break;
		
		case 'l':
			var=levinson(ac, x, lx, p);	//x is rx			
			MDft (Sp, ac, Nfft, p+1);					
			for(i=0; i<Nfft; i++)
				Sp[i]=var/(Sp[i]*Sp[i]);			
			
			break;
	}
		

}


 
