function gmm2hmmPCA ( inputGMMFile, outputHMMFile, pcaModel, trainList, method, byteOrder )

if nargin < 6
    byteOrder= 'l'; %Big-endian
end

if nargin < 5
    method= 'soft';
end

switch method
    case 'hard'
        compute= @hardTransPComputation;
    case 'soft'
        compute= @softTransPComputation;
end

disp( '1.- Reading the GMM and PCA files' );
disp( '======================' );
load( inputGMMFile, 'mix' );
load( pcaModel, 'model' );
gmm= mix;

disp( '2.- Computing the transition probabilities' );
disp( '=============================' );
mix= compute( gmm, model, trainList, byteOrder );

disp( '3.- Saving the HMM with the transition probabilities' );
disp( '====================' );
save( outputHMMFile, 'mix' );


%------------------------------------------------------------------------
function hmm= hardTransPComputation( gmm, pcaModel, trainList, byteOrder )

K= gmm.ncentres;
hmm= gmm;
logmix= precomputeLogMix( gmm );
A= zeros(K, K); %Transition probabilities
p= zeros(K,1); %Likelihood for every Gaussian

dimPCA= gmm.nin;
P= pcaModel.eigVecs(:,1:dimPCA)'; %Projection matrix with the dim principal components

%Process the training files
list= textread( trainList, '%s' );
nfiles= length( list );
for f=1:nfiles
    disp( ['Processing file ' num2str(f) '/' num2str(nfiles) ' ...'] );
    
    %Read the original file, apply an sliding window and project to the PCA
    %space
    data= readhtk( list{f}, byteOrder );    
    data= enventana3( data, pcaModel.winLen, pcaModel.winShift );    
    data= bsxfun( @minus, data, pcaModel.mean );
    data= P*data;
    [params T]= size(data);
   
    k_t_1= -1; %Most likely gaussian for the previous frame
    for t=1:T
        x= data(:,t);

        %Compute the likelihood of every Gaussian
        for k=1:K
            p(k)= logPosteriorProb( logmix, k, x );
        end
        
        [pmax, k_t]= max( p );
        if k_t_1 ~= -1
            A(k_t_1, k_t)= A(k_t_1, k_t) + 1;
        end

        k_t_1= k_t;
    end
end

%Normalize the transition probabilities
hmm.transp= bsxfun(@times, A, 1./sum(A,2));

%If there exists any unobservable cluster, then the transtion probabilities are
%set to the a priori probabilities
[rows cols]= find( isnan(hmm.transp) );
rows= unique( rows );
if ~isempty(rows) 
    hmm.transp(rows,:)= repmat( gmm.priors, length(rows), 1 );
end


%------------------------------------------------------------------------
function hmm= softTransPComputation( gmm, pcaModel, trainList, byteOrder )

K= gmm.ncentres;
hmm= gmm;
logmix= precomputeLogMix( gmm );
A= zeros(K, K); %Transition probabilities
p_t_1= zeros(K,1); %Probability of every Gaussian for the previous frame
p_t= zeros(K,1); %Probability of every Gaussian for the current frame

dimPCA= gmm.nin;
P= pcaModel.eigVecs(:,1:dimPCA)'; %Projection matrix with the dim principal components

%Process the training files
list= textread( trainList, '%s' );
nfiles= length( list );
for f=1:nfiles
    disp( ['Processing file ' num2str(f) '/' num2str(nfiles) ' ...'] );
    %Read the original file, apply an sliding window and project to the PCA
    %space
    data= readhtk( list{f}, byteOrder );    
    data= enventana3( data, pcaModel.winLen, pcaModel.winShift );    
    data= bsxfun( @minus, data, pcaModel.mean );
    data= P*data;
    [params T]= size(data);
 
    for t=1:T
        x= data(:,t);

        %Compute the likelihood of every Gaussian
        for k=1:K
%             p_t(k)= posteriorProb( gmm, k, x );
            p_t(k)= posteriorProbFast( logmix, k, x );
        end
        p_t= p_t/sum(p_t);
        
        if t>1
            A= A + p_t_1*p_t';
        end

        p_t_1= p_t;
    end
end

%Normalize the transition probabilities
hmm.transp= bsxfun(@times, A, 1./sum(A,2));

%If there exists any unobservable cluster, then the transtion probabilities are
%set to the a priori probabilities
[rows cols]= find( isnan(hmm.transp) );
rows= unique( rows );
if ~isempty(rows) 
    hmm.transp(rows,:)= repmat( gmm.priors, length(rows), 1 );
end


%------------------------------------------------------------------------
function p= posteriorProb( gmm, k, x )
mean= gmm.centres(k,:)';
if strcmp(gmm.covar_type,'diag')
    s= 1./sqrt(gmm.covars(k,:)');
    z= s.*(x-mean);
    p= prod( normpdf(z).*s );    
else
    S= gmm.covars(:,:,k);
    p= mvnpdf( x, mean, S );
end
p= p * gmm.priors(k);


%------------------------------------------------------------------------
function p= posteriorProbFast( gmm, k, x )
mean= gmm.centres(k,:)';
if strcmp(gmm.covar_type,'diag')
    inv_s= sqrt(gmm.invCovars(k,:)');
    z= inv_s.*(x-mean);
    p= prod( normpdf(z).*inv_s );    
else
    invS= gmm.invCovars(:,:,k);
    z= (x-mean);
    p= exp( -0.5*(z'*invS*z + gmm.gconst(k)) );
end
p= p * gmm.priors(k);


%------------------------------------------------------------------------
function p= logPosteriorProb( gmm, k, x )
mean= gmm.centres(k,:)';
if strcmp(gmm.covar_type,'diag')
    inv_s= gmm.invCovars(k,:)';
    z= x-mean;
    p= -0.5*(gmm.gconst(k) + sum( z.*z.*inv_s ));
else
    invS= gmm.invCovars(:,:,k);
    z= (x-mean);
    p= -0.5*(gmm.gconst(k) + z'*invS*z);
end
p= p + gmm.logPriors(k);


%------------------------------------------------------------------------
function logMix= precomputeLogMix ( mix )
K= mix.ncentres;
P= mix.nin;
LN_2PI= P*1.837877066409345; %P*log(2*pi);

logMix= mix;
logMix.logPriors= log( mix.priors );
logMix.gconst= zeros( 1, K );
if strcmp(mix.covar_type,'diag')
    logMix.invCovars= zeros(K,P);
else
    logMix.invCovars= zeros(P, P, K);
end

for k= 1:K
    if strcmp(mix.covar_type,'diag')
        s= mix.covars(k,:);
        detS= sum( s );
        logMix.invCovars(k,:)= 1./s;
    else
        S= mix.covars(:,:,k);
        detS= log(det(S));
        logMix.invCovars(:,:,k)= inv(S);
    end
    logMix.gconst(k)= LN_2PI + detS;
end
