%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
%  INFOS 
%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
% Author: Hannes Pessentheiner 
% Date  : 05. July 2011 
% Update: 22. January 2012 
%
% Function: 
%   [buffer, angles, vad, numFrames, SigOut, SigInpPow, NoiInpPow, SigOutPow, NoiOutPow, iGSINR, GAIN] = prcData(params, buffer)  
%
% Function Description: 
%   This function reads the microphone array geometry from a dat-file 
%   and computes and writes the weights using different 'data dependent' 
%   or 'data independent' beamformers. It also computes the output data of
%   the selected beamformer.
%
% Input-Parameters: 
%   (struct) params         ... structure containing global parameters 
%   (struct) buffer         ... buffer 
%
% Output-Parameters: 
%   (struct) buffer         ... buffer
%   (array)  angles         ... vector containing angles  
%   (array)  vad            ... array containing information about voice activity detection
%   (int)    numFrames      ... number of frames 
%   (array)  SigOut         ... signal output (mono) 
%   (array)  SigInpPow      ... signal input power  (frame-wise) 
%   (array)  NoiInpPow      ... noise  input power  (frame-wise) 
%   (array)  SigOutPow      ... signal output power (frame-wise) 
%   (array)  NoiOutPow      ... noise  output power (frame-wise) 
%   (float)  iGSINR         ... improvment in GSINR 
%   (float)  GAIN           ... gain of the enhanced file 



function [buffer, angles, vad, numFrames, SigOut, SigInpPow, NoiInpPow, SigOutPow, NoiOutPow, iGSINR, GAIN] = prcData(params, buffer) 
%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
%  DEFINITIONS
%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
%  Set General Parameters: 
%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
% Introduction of  parameters like, e.g., sound velocity, frame size, 
% sampling frequency, source distances, angles, etc.
numMics = params.Basic.NumMics; 
numBins = params.Basic.NumBins; 
totMics = params.Basic.TotMics; 


%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
%  Read Audio Files: 
%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
for iMic=1:numMics 
    [sig(:,iMic), fsSrc] = wavread([params.Audio.FilePathSrc, params.Audio.FileNameSrc{iMic}]);
    
    if params.Basic.SimData == 0
        disp(['    Ch' num2str(1+(iMic-1)*totMics/numMics) ': Calibration considered! Computation with real data.']); 
        [sig(:,iMic)] = sig(:,iMic) * params.Array.Calibration( 1+(iMic-1)*totMics/numMics );
    else 
        disp(['    Ch' num2str(1+(iMic-1)*totMics/numMics) ': No calibration considered! Computation with synthetic data.']); 
    end
    
end
numSamples = min(size(sig,1),params.Audio.LenFile * fsSrc);


%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
%  Additional Memory Allocation: 
%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
% General Parameters 
vad        = zeros(numSamples,1);       % voice activity detection 
ProgCur    = 0;                         % progression 
numFrames  = floor(numSamples/numBins);

% Time Domain
SigOut     = zeros(numBins,1); 
SigOutPre  = zeros(numBins*2,1); 
SigInpPow  = zeros(numMics,numFrames);
SigOutPow  = zeros(numFrames,1); 
NoiInpPow  = zeros(numMics,numFrames); 
NoiOutPow  = zeros(numFrames,1); 
% Frequency Domain 
SpecNoi    = zeros(numMics,numBins); 
SpecInp    = zeros(numMics,numBins); 
SpecOut    = zeros(numBins,1); 
SpecOutPre = zeros(numBins,1); 
% Memory allocation for covariance matrix. 
buffer.Noise.MatCov = zeros(numMics,numMics,numBins ,'single'); 



%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
%  PROCESS
%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
% To avoid a long processing time, calculate weights only once, 
% if beamformer is data-independent
if params.Beam.Type(2) == 0
    disp('    Non-adaptive computation of coefficients started.')
        [w, angles] = cmpWeights(params);
        save([params.Basic.DirPathTemp '/w'],'w'); 
    disp(['--> Done.' char(10)])
end


% Process audio data frame-wise, which enables one to implement adaptive
% and/or data-dependent beamformers, e.g., the Generalized Sidelobe Canceller. 
if params.Beam.Type(2) == 0
    disp('--> Processing data with non-adaptive coefficients.')
end


% Consider 'numFrames-1' because of FrameCurL. 
for iFrame = 1:numFrames-1 
    % Compute a sample index. 
    SIdx = (iFrame-1)*numBins+1; 
    
    
    % Compute and display the progression (abbr.: prog). 
    ProgPre = ProgCur; 
    ProgCur = floor(SIdx/numSamples*100);
    if ProgPre ~= ProgCur
        disp(['    Progress: ' num2str(ProgCur) ' %'])
    end    
    
    
    % Determine samples of single/double frame (used for overlap). 
    FrameCurS = SIdx:SIdx+  numBins-1; % single frame (short) 
    FrameCurL = SIdx:SIdx+2*numBins-1; % double frame (long)
    
    
    % Convert Signal into f-Domain:  
    for iMic = 1:numMics
        SpecInp(iMic,:) = cmpSpecSignal(sig(FrameCurL,iMic));        
    end

    
    % Calculate weights, if beamformer is data-dependent. Therefore, flag
    % has to be set to '1', otherwise '0'. 
    if params.Beam.Type(2) == 0
        [~, curBeam] = min(abs(angles - params.Array.MRA));         
        SpecOut(:) = sum(squeeze(w(curBeam,:,:)) .* SpecInp);
    else
        [w, angles, SpecNoi, buffer] = cmpWeights(params, buffer, SIdx, SpecInp, SpecNoi);        
        save([params.Basic.DirPathTemp '/w' num2str(iFrame)],'w'); 
        
        [~, curBeam] = min(abs(angles - params.Array.MRA));         
        SpecOut(:) = sum(squeeze(w(curBeam,:,:)) .* SpecInp);
    end
    
    
    % Convert signal into t-Domain for saving it as a wav-file. 
    [SigOut(FrameCurS), SigOutPre] = cmpTimeSignal(SpecOut, SigOutPre);

    
    % Measure the RMS values for all input channels and the output. Proper
    % log-files have to be provided. Usually, some changes have to be 
    % considered. The voice activity detection depends on the log-files.
    % Voice states, e.g., silence, voiced or unvoiced speech, are labeled
    % by hand.
    SilPosCur = length(params.Audio.VoiceState1(SIdx >= params.Audio.VoiceState1)); 
    SilPosNxt = length(params.Audio.VoiceState1(SIdx + 3*numBins >= params.Audio.VoiceState1)); 
    strCur    = params.Audio.VoiceState2{SilPosCur}; 
    strNxt    = params.Audio.VoiceState2{SilPosNxt};  

    % Consider the next voice state to compensate small labeling errors. 
    if strcmp(strCur,'sil') && strcmp(strNxt,'sil') 
        NoiOutRMS = cmpRMS(SpecOut);              % RMS 
        NoiOutPow(iFrame) = NoiOutRMS^2;          % Power 
        for iMic = 1:numMics
            NoiInpRMS = cmpRMS(SpecInp(iMic,:));  % RMS 
            NoiInpPow(iMic,iFrame) = NoiInpRMS^2; % Power 
        end
    else
        vad(SIdx : SIdx+numBins-1,1) = 1;         % Voice Activity Detection 
        SigOutRMS = cmpRMS(SpecOut);              % RMS 
        SigOutPow(iFrame) = SigOutRMS^2;          % Power 
        for iMic = 1:numMics
            SigInpRMS = cmpRMS(SpecInp(iMic,:));  % RMS
            SigInpPow(iMic,iFrame) = SigInpRMS^2; % Power 
        end
    end
    
    
    % Save previous spectrum for overlap. 
    SpecOutPre(:) = SpecOut;
end
SigOut(FrameCurS) = SigOutPre(numBins+1:2*numBins);


%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
%  Compute SINR:  
%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
[iGSINR, GAIN] = cmpGSINR(params, SigInpPow, NoiInpPow, SigOutPow, NoiOutPow); 


%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
%  Write Nearest-Mic File:  
%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
% Now, store a lowpass-filtered and down-sampled reference file for the
% computation of objective measurements (here: from 48000 Hz to 16000 Hz). 
wavwrite(resample( sig(1:length(SigOut),cmpMicNearest(params)),1,fsSrc/params.Basic.fsOut), params.Basic.fsOut, [params.Audio.FilePathRec, params.Audio.FileNameRec, '_nm.wav'] );


%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
%  Write Single-Speaker File (nearest-mic):  
%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
% Now, store a lowpass-filtered and down-sampled reference file for the
% computation of objective measurements (here: from 48000 Hz to 16000 Hz). 
[sigSingleSpeaker, fsSrc] = wavread([params.Audio.FilePathSrcSingle, params.Audio.FileNameSrcSingle{cmpMicNearest(params)}]);
wavwrite(resample( sigSingleSpeaker(1:length(SigOut)),1,fsSrc/params.Basic.fsOut), params.Basic.fsOut, [params.Audio.FilePathRec, params.Audio.FileNameSrcSingle{cmpMicNearest(params)}] );


%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
%  Write Output File:  
%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
% Now, store a lowpass-filtered and down-sampled output file for the
% computation of objective measurements (here: from 48000 Hz to 16000 Hz). 
wavwrite(resample(SigOut,1,fsSrc/params.Basic.fsOut), params.Basic.fsOut, [params.Audio.FilePathRec, params.Audio.FileNameRec]);