function [WeightsArray, isDependent] = rlsfi(params, ~, CapturingResponseLinArray, varargin)
% FUNCTION DESCRIPTION: 
%   Determines the coefficients of a data-independent robust 
%   least squares frequency invariant beamformer based on convex optimization. 
% 
% INPUT PARAMETERS: 
%   (struct) params.Ma.Array.nMicTotal     ... number of microphones      
%            params.Ma.Basic.nAzim         ... number of azimuth angles        
%            params.Ma.Basic.dAngleAzim    ... differential azimuth angle          
%            params.Ma.Beam.AngleIndexAzim ... azimuth angle index of steering direction          
%            params.Ma.Beam.AngleIndexElev ... elevation angle index of steering direction            
%   (array)  CapturingResponseArray        ... capturing response 
% 
% OUTPUT PARAMETERS: 
%   (array)  WeightsArray ... calculated weights 
%   (bool)   isDependent  ... information about data-dependency of beamformer 
%
% UPDATE: 09. October 2012, Hannes Pessentheiner 

%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
%  HEADER:   
%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
if nargin < 2
    error('ERROR: Not enough input parameters. Function terminated.');
end

%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
%  DEFINITIONS 
%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
isDependent    = 0;      % data-independent beamformer  

OPT_BW         = 20;     % Beamwidth (deg).
OPT_GAMMA      = 0.001;  % Scaling factor for the RLSFI-BF.

G = double(squeeze(CapturingResponseLinArray(params.Ma.Beam.AngleIndexElev, :                            , :))); 
d = double(squeeze(CapturingResponseLinArray(params.Ma.Beam.AngleIndexElev, params.Ma.Beam.AngleIndexAzim, :))); 
d = d(:); 
b = zeros(params.Ma.Basic.nAzim, 1); 

WeightsArray = zeros(params.Ma.Array.nMicTotal, 1);   

%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
%  PROCESS 
%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
nBwIndicesAzim      = floor(OPT_BW / params.Ma.Basic.dAngleAzim); 
b(1:nBwIndicesAzim) = 1; 
b = circshift(b,[floor(params.Ma.Beam.AngleIndexAzim-1 - nBwIndicesAzim/2),0]); 

% Convex programming starts here. 
% NOTE: Values have to be available in double precision! 
cvx_begin quiet; 
    variable WeightsArray(params.Ma.Array.nMicTotal,1) complex; 
    minimize (sum_square_abs(G*WeightsArray-b)); 
    subject to 
        (WeightsArray.') * d == 1;
           WeightsArray' * WeightsArray <= 1/OPT_GAMMA; 
cvx_end; 
% Convex programming ends here. 

end