function [WeightsArray, isDependent] = mnscvx(params, ~, CapturingResponseLinArray, varargin)
% FUNCTION DESCRIPTION: 
%   Determines the coefficients of a convex optimization-based 
%   beamformer with multiple null-steering. 
% 
% INPUT PARAMETERS: 
%   (struct) params.Ma.Array.nMicTotal     ... number of microphones      
%            params.Ma.Basic.nAzim         ... number of azimuth angles        
%            params.Ma.Basic.dAngleAzim    ... differential azimuth angle    
%            params.Ma.Basic.dAngleElev    ... differential elevation angle 
%            params.Ma.Beam.AngleIndexAzim ... azimuth angle index of steering direction          
%            params.Ma.Beam.AngleIndexElev ... elevation angle index of steering direction            
%   (array)  CapturingResponseArray        ... capturing response 
%
% OUTPUT PARAMETERS: 
%   (array)  WeightsArray ... calculated weights 
%   (bool)   isDependent  ... information about data-dependency of beamformer 
%
% UPDATE: 09. October 2012, Hannes Pessentheiner 

%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
%  HEADER:   
%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
if nargin < 2
    error('ERROR: Not enough input parameters. Function terminated.');
end

%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
%  DEFINITIONS 
%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
isDependent    = 0;      % data-independent beamformer  
OPT_BW         = 30;     % Beamwidth (deg).                         % ORIG: 20
OPT_GAMMA      = 0.1;  % Scaling factor for the RLSFI-BF.         % IROG: 0.001

% NOTE: The matrix b considers 360 deg elevation instead of 180 deg.
G = double(CapturingResponseLinArray); 
d = double(squeeze(CapturingResponseLinArray(params.Ma.Beam.AngleIndexElev, params.Ma.Beam.AngleIndexAzim, :))); 
G = G(:,:); % Matrices are cascaded horizontally! 
b = zeros(params.Ma.Basic.nElev * 2, params.Ma.Basic.nAzim); 
WeightsArray = zeros(params.Ma.Array.nMicTotal, 1);   

%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
%  PROCESS 
%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
nBwIndicesAzim = floor(OPT_BW / params.Ma.Basic.dAngleAzim); 
nBwIndicesElev = floor(OPT_BW / params.Ma.Basic.dAngleElev); 

b(1:nBwIndicesElev, 1:nBwIndicesAzim) = 1; 

% Now, shift the ones to the right positions. 
b = circshift(b                                                        , ...   
              [                                                          ...
               floor(params.Ma.Beam.AngleIndexElev - nBwIndicesElev/2), ...
               floor(params.Ma.Beam.AngleIndexAzim - nBwIndicesAzim/2)  ...
              ]                                                          ...
             ); 

% Reduce matrix size by mapping the values at elev > 180 deg 
% to 0 < elev < 180. An azimuth of 0 deg and elevation of 181 deg 
% corresponds to an azimuth of 180 deg and elevation of 179 deg. 
% NOTE: b = [b1 b2; b3 b4] -> bNew = [b1+flipud(b4), b2+flipud(b3)]
b3 = b(params.Ma.Basic.nElev+1 : end, 1         : params.Ma.Basic.nAzim/2);                          
b4 = b(params.Ma.Basic.nElev+1 : end, params.Ma.Basic.nAzim/2+1 : end    );                                          

bTemp(1:params.Ma.Basic.nElev, 1:params.Ma.Basic.nAzim      ) = b(1:params.Ma.Basic.nElev, 1:params.Ma.Basic.nAzim); 
bTemp(1:params.Ma.Basic.nElev, 1:params.Ma.Basic.nAzim/2    ) = flipud(b4) + bTemp(1:params.Ma.Basic.nElev, 1:params.Ma.Basic.nAzim/2    );
bTemp(1:params.Ma.Basic.nElev, params.Ma.Basic.nAzim/2+1:end) = flipud(b3) + bTemp(1:params.Ma.Basic.nElev, params.Ma.Basic.nAzim/2+1:end);

b = bTemp; 

% Convex programming starts here.
% HINT: Use Frobenius-Norm! 
cvx_begin quiet; 
    variable WeightsArray(params.Ma.Array.nMicTotal,1) complex; 
    minimize (norm(G*(kron(WeightsArray, eye(params.Ma.Basic.nAzim)))-b,'fro')); 
    subject to 
        (WeightsArray.') * d == 1;
           WeightsArray' * WeightsArray <= 1/OPT_GAMMA; 
cvx_end; 
% Convex programming ends here. 

end
