function [SystemResponseAbsArray, SystemResponseLogArray] = computeSystemResponse(params, CapturingResponseArray, WeightsArray) 
% FUNCTION DESCRIPTION: 
%   Determines the 2 or 3 dimensional system response 
%   including the directivity pattern. According to 'meshgrid()' oder 
%   'gradient()' the number of columns of the system response corresponds 
%   to the number of elevation angles, whereas the number of rows 
%   corresponds to the number of azimuth angles. 
% 
% INPUT PARAMETERS: 
%   (struct) params.Ma.Basic.nBin                ... number of bins 
%            params.Ma.Basic.nAzim               ... number of azimuth angles 
%            params.Ma.Basic.nElev               ... number of elevation angles 
%            params.Ma.Array.nMicTotal           ... number of microphones 
%            params.Global.Display.isProgression ... display progression 
%   (array)  CapturingResponseArray              ... capturing response linear 
%   (array)  WeightsArray                        ... beamformer weights 
%   
% OUTPUT PARAMETERS: 
%   (array) SystemResponseAbsArray ... system response absolute values  
%   (array) SystemResponseLogArray ... system response logarithmical 
%
% UPDATE: 09. October 2012, Hannes Pessentheiner 

%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
%  HEADER:  
%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
fprintf('\nCOMPUTE System Response ... '); 

if nargin < 3
    error('ERROR: Not enough input parameters. Function terminated.');
end

%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
%  DEFINITIONS:  
%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
CapturingResponseWeightedArray  = zeros(params.Ma.Basic.nElev * params.Ma.Basic.nBin/2, ...
                                        params.Ma.Basic.nAzim                         , ...
                                        params.Ma.Array.nMicTotal                     , ...
                                        'single');                                                

%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
%  PROCESS: 
%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
for iMic = 1:params.Ma.Array.nMicTotal       
    computeProgression(iMic                                , ...
                       params.Ma.Array.nMicTotal           , ...
                       params.Global.Display.isProgression);
    
    % VECTOR (1xK) --> MATRIX (MxK) --> MATRIX (MKxN)
    w = WeightsArray(iMic,:); 
    w = w(:).'; 
    w = w(ones(params.Ma.Basic.nElev, 1),:);
    w = w(:); 
    w = w(:,ones(params.Ma.Basic.nAzim,1)); 
    
    % Put this in a separate function. 
    % Use 3D-representation due to the summation "sum(MAT,3)".   
    % MATRIX (MKxNxL)
    CapturingResponseWeightedArray(:,:,iMic) = w .* CapturingResponseArray(:,:,iMic); 
end

% MATRIX (MKxN)
SystemResponseLinArray = sum    (CapturingResponseWeightedArray,3);                                
SystemResponseLinArray = double (SystemResponseLinArray);
SystemResponseAbsArray = abs    (SystemResponseLinArray); 
SystemResponseLogArray = 20 * log10(abs(SystemResponseLinArray)); 
SystemResponseLogArray(SystemResponseLogArray <= -60) = -60; 

clear SystemResponseLinArray; 

end

%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
%  INFO:   
%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
% MATRIX/VECTOR DIMENSIONS: 
%   K ... number of frequency bins 
%   L ... number of microphones 
%   M ... number of elevation angles 
%   N ... number of azimuthal angles 