function c = computeSoundVelocity(temperature)
% FUNCTION DESCRIPTION: 
%   Computes the speed of sound with respect to the temperature
%   in degree Celsius. 
% 
% INPUT PARAMETERS: 
%   (float) temperature ... temperature in degree Celsius 
% 
% OUTPUT PARAMETERS: 
%   (float) c ... sound velocity 
% 
% NOTE: The approximate speed of sound in dry (0% humidity) air, in meters 
% per second, at temperatures near 0 degree Celsius, can be calculated from
% the practical formula for dry air: c = 20.0457 * sqrt(temperature + 273.15)
% This equation is correct to a much wider temperature range, but still 
% depends on the approximation of heat capacity ratio being independent of 
% temperature, and for this reason will fail, particularly at higher temperatures. 
% It gives good predictions in relatively dry, cold, low pressure conditions, 
% such as the Earth's stratosphere. The equation fails at extremely low pressures 
% and short wavelengths, due to dependence on the assumption that the wavelength 
% of the sound in the gas is much longer than the average mean free path between 
% gas molecule collisions (Wikipedia, Speed of Sound, 27. July 2012). 
%
% UPDATE: 24. October 2012, Hannes Pessentheiner 

%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
%  HEADER:   
%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
if nargin < 1
    error('ERROR: Not enough input parameters. Function terminated.');
end

%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
%  PROCESS:   
%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
c = 20.0457 * sqrt(temperature + 273.15);

end