function [DirectivityPatternLinArray, DirectivityPatternLogArray] = computeMicrophoneDirectivity(params, MicType)
% FUNCTION DESCRIPTION: 
%   Computes the microphone directivity. 
%
% INPUT PARAMETERS: 
%   (struct) params  ... structure containing global parameters
%   (int)    MicType ... microphone type 
% 
% OUTPUT PARAMETERS: 
%   (array)  DirectivityPatternLinArray ... directivity pattern linear 
%   (array)  DirectivityPatternLogArray ... directivity pattern logarithmical
%
% AUTHOR:   Hannes Pessentheiner 
% UPDATE:   20. April 2012 


%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
%  HEADER: 
%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
if nargin < 2
    error('ERROR: Not enough input parameters. Function terminated.');
end

%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
%  DEFINITIONS 
%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
OPT_TORAD = pi/180; 
OPT_DELTA = 0.005; % microphone length (m)

DirectivityPatternLinArray = zeros(params.Ma.Basic.nBins, params.Ma.Basic.nAzims); 

%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
%  PROCESS  
%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
for iAzim=1:params.Ma.Basic.nAzims
    % Ideal gradient microphone (far-field assumption). 
    DirectivityPatternArray = 1.0 - exp(-1i * 2*pi*params.Ma.Audio.fArray * OPT_DELTA / params.Ma.Basic.c * cos(OPT_TORAD * params.Ma.Basic.AzimArray(iAzim)));
    DirectivityPattern1000  = 1.0 - exp(-1i * 2*pi*1000                   * OPT_DELTA / params.Ma.Basic.c);
    
    % Normalize directivity pattern. 
    DirectivityPatternArray = DirectivityPatternArray / DirectivityPattern1000; 

    % Consider compensation filter (1st order lowpass filter) to obtain a 
    % constant frequency response for frequencies above 100 Hz. The 
    % directivity pattern defined above yields a linear (increasing) frequency 
    % response due to the different wavelength. 
    tau = 1e-3;     % Time-constant (tau=RC) 
    C   = 33e-6;    % Capacitor
    R   = tau / C;  % Resistor 

    % Lowpass-filter impedance. 
    Z1000  = - 1i  / (2*pi*1000 * C); 
    ZArray = - 1i ./ (2*pi*params.Ma.Audio.fArray * C);    
    ZArray(params.Ma.Audio.fArray < 1e-10) = - 1i * 1e-10;

    % Filter transfer function. 
    LowPassFilter1000  = Z1000   / (Z1000  + R);
    LowPassFilterArray = ZArray ./ (ZArray + R); 

    % Normalize lowpass filter. 
    LowPassFilterArray = LowPassFilterArray / LowPassFilter1000;

    % Resulting microphone directivity pattern. 
    DirectivityPatternLinArray(:,iAzim) = DirectivityPatternArray .* LowPassFilterArray;
end

switch MicType
    case 0          % omnidirectional
        a = 1.00;
        b = 0.00;
    case 1          % subcardioid 
        a = 0.75;
        b = 0.25;
    case 2          % cadrioid
        a = 0.50;
        b = 0.50;
    case 3          % supercardioid
        a = 0.37;
        b = 0.63;
    case 4          % hypercardioid
        a = 0.25;
        b = 0.75;
    case 5          % pressure gradient microphone
        a = 0.00;
        b = 1.00;
end

DirectivityPatternLinArray =  a + b * DirectivityPatternLinArray; 
DirectivityPatternLogArray = 20*log10(DirectivityPatternLinArray); 

end