function [CapturingResponseLinArray] = computeCapturingResponse(params) 
% FUNCTION DESCRIPTION: 
%   Determines the 2 or 3 dimensional capturing response. 
%   According to 'meshgrid()' oder 'gradient()' the number of columns of the 
%   capturing response corresponds to the number of elevation angles, 
%   whereas the number of rows corresponds to the number of azimuth angles. 
% 
% INPUT PARAMETERS: 
%   (struct) params.Ma.Basic.nBin                ... number of bins
%            params.Ma.Basic.nAzim               ... number of azimuth angles 
%            params.Ma.Basic.nElev               ... number of elevation angles 
%            params.Ma.Basic.AzimArray           ... azimuth angles 
%            params.Ma.Array.nMicTotal           ... number of microphones 
%            params.Ma.Array.Mic.ElevArray       ... microphone elevation angles 
%            params.Ma.Array.Mic.rArray          ... microphone radii 
%            params.Global.Audio.fArray          ... frequencies 
%            params.Global.Display.isProgression ... display progression
%   
% OUTPUT PARAMETERS: 
%   (array) CapturingResponseLinArray ... capturing response linear  
%
% FORMULA: 
%   h(m,n,s1,s2) = 2*pi*f_m / c * |r_n| * {sin(theta_s1) * sin(theta_n) * cos(phi_s1 - phi_n) + cos(theta_s1) * cos(theta_n)}
%
% UPDATE: 09. October 2012, Hannes Pessentheiner 

%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
%  HEADER:  
%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
fprintf('\nCOMPUTE Capturing Response ... '); 

if nargin < 1
    error('ERROR: Not enough input parameters. Function terminated.');
end

%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
%  DEFINITIONS:  
%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
OPT_TORAD = pi/180; 

CapturingResponseLinArray = zeros(params.Ma.Basic.nElev * params.Ma.Basic.nBin/2, ...
                                  params.Ma.Basic.nAzim                         , ...
                                  params.Ma.Array.nMicTotal                     , ...
                                  'single');                           

%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
%  PROCESS
%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~                       
% Faster than using repmat(). 
SinElevArray     = sin(params.Ma.Basic.ElevArray(:) * OPT_TORAD); % d1
SinElevArray     = SinElevArray(:,ones(1,params.Ma.Basic.nBin/2)); 
SinElevArray     = SinElevArray(:); 
SinElevArray     = SinElevArray(:,ones(params.Ma.Basic.nAzim,1)); 

CosElevArray     = cos(params.Ma.Basic.ElevArray(:) * OPT_TORAD); % d3
CosElevArray     = CosElevArray(:,ones(1,params.Ma.Basic.nBin/2)); 
CosElevArray     = CosElevArray(:); 
CosElevArray     = CosElevArray(:,ones(params.Ma.Basic.nAzim,1)); 

fArray           = +1i*2*pi/params.Ma.Basic.c * params.Global.Audio.fArray(1:end/2); % f1
fArray           = fArray(:).'; 
fArray           = fArray(ones(params.Ma.Basic.nElev,1),:); 
fArray           = fArray(:); 
fArray           = fArray(:,ones(params.Ma.Basic.nAzim,1));

SinMicThetaArray = sin(params.Ma.Array.Mic.ElevArray * OPT_TORAD); % d4
CosMicThetaArray = cos(params.Ma.Array.Mic.ElevArray * OPT_TORAD); % d5

DistMicArray     = params.Ma.Array.Mic.rArray; % rn
       
for iMic = 1:params.Ma.Array.nMicTotal    
    computeProgression(iMic                                , ...
                       params.Ma.Array.nMicTotal           , ...
                       params.Global.Display.isProgression);
    
    % Here use repmat() for simplified computations. 
    % VECTOR (Nx1) --> MATRIX (NxM) --> MATRIX (MxN) --> MATRIX (MKxN) 
    CosAzimMicPhiArray = cos((params.Ma.Basic.AzimArray(:) - params.Ma.Array.Mic.AzimArray(iMic)) * OPT_TORAD);
    CosAzimMicPhiArray = CosAzimMicPhiArray(:,ones(1,params.Ma.Basic.nElev)).'; 
    CosAzimMicPhiArray = repmat(CosAzimMicPhiArray,[params.Ma.Basic.nBin/2 1]); 

    % Put this in a separate function. Use 3D-representation due to the summation "sum(MAT,3)". 
    % MATRIX (MKxNxL) 
    CapturingResponseLinArray(:,:,iMic) =  exp(fArray                   ...
                                          * DistMicArray(iMic)          ...
                                         .* (                           ...
                                               SinElevArray             ...
                                             * SinMicThetaArray(iMic)   ...
                                            .* CosAzimMicPhiArray       ...
                                             + CosElevArray             ...
                                             * CosMicThetaArray(iMic)   ...
                                            )                           ...
                                         );
end

end

%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
%  INFO:   
%~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
% MATRIX/VECTOR DIMENSIONS: 
%   K ... number of frequency bins 
%   L ... number of microphones 
%   M ... number of elevation angles 
%   N ... number of azimuthal angles 